###################################################
### duque_stevens_rcode.R
### Created by Juan F. Duque on 28 August 2015 (jfduque89@gmail.com),
###	 finalized on 16 Aug 2016
### Summary: This script analyzes and produces plots for dominance and 
### 	food sharing data pinyon jays.
### Instructions: Place this file and the data files in the main directory.
### 	Create a folder called "figures". Set the R working directory to the 
### 	main directory.  At the R command prompt, type
### 	> source("duque_stevens_rcode.R")
### 	This will run the script, adding all of the calculated variables to the
### 	workspace and saving figures in the figures directory.
### Uses: This script can be reproduced and modified for personal and scientific use.
### Data files:
###  duque_stevens_reciprocity_data.csv
### 	squad - squad number
### 	date - date of session
### 	daily - designates whether sessions occurred in morning or afternoon if  
### 	 dyads experienced two sessions within a day or single if dyads only experienced  
### 	 one session that day (relevant for reciprocity where successive sessions  
### 	 occurred either the same day or across several days)
###   dyad - subject identification numbers for dyad
### 	session - corresponding session for that dyad of birds; increases by  
### 	 one each time that dyad is tested
### 	donor - the individual bird who could be a potential donor (was given access  
### 	 to food cup during session)
### 	recipient - the individual bird who could be a potential recipient (was NOT given  
### 	 access to food cup during session)
### 	shares - number of sharing events (food transfers from donor to recipient)  
### 	 observed for that specific session
### 	any_shares - scores whether any sharing occurred (0 = no sharing occurred, 1 = 
###    sharing occurred)
### 	direct_recip - scores whether direct reciprocity occurred. NA if donor in current 
###    session was also the donor in the previous session with that same partner; 
### 	 if the donor and recipient roles reversed fromthe previous session to the 
### 	 current one, then: 0 if no sharing occurred in that previous session, 
### 	 1 if sharing occurred in that previous session
### 	general_recip - scores whether generalized reciprocity occurred. Same as above 
###    except for generalized reciprocity partner identity does not matter;  
### 	 therefore the “previous session” is merely the previous session the current 
### 	 donor was tested, regardless of partner. NA if donor in current session was  
### 	 also the donor in its previous session; if the current donor was a recipient
### 	 in its previous session (i.e., reversed roles), then: 0 if no sharing occurred  
### 	 in that previous session, 1 if sharing occurred in that previous session
###  duque_stevens_dominance_data.csv
### 	squad - squad number
###   dyad - subject identification numbers for dyad
### 	dominance - relative dominance score. 0 if first bird from dyad (e.g., bird X 
### 	 from dyad XZ) did NOT get food before the second bird (Z) or 1 if first bird  
### 	 from dyad did get food first (e.g., bird X obtained food prior to bird Z from  
### 	 dyad XZ); NA if no dominance encounter was tested that day
###################################################

##########################
## Load libraries
##########################
rm(list = ls())       # clear all variables

library(igraph)       # needed for network graph
library(lattice)      # needed for lattice graphs
library(latticeExtra) # needed for combining separate lattice plots
library(lme4)         # needed for lmer

col.blind <- c("#0072B2", "#D55E00")  # define color-blind safe colors
               
##########################
## Load and prepare data
##########################
## Sharing data
sharing_data <- read.csv("duque_stevens_reciprocity_data.csv")	# input sharing data
sharing_data$dyad <- as.factor(sharing_data$dyad)           		# convert dyad to factor
sharing_data$donor <- as.factor(sharing_data$donor)         		# convert donor to factor
sharing_data$recipient <- as.factor(sharing_data$recipient) 		# convert recipient to factor

sharing_data1 <- subset(sharing_data, squad == 1) # subset out squad 1 data
sharing_data2 <- subset(sharing_data, squad == 2) # subset out squad 2 data

## Dominance data
dominance_data <- read.csv("duque_stevens_dominance_data.csv", header = TRUE)  # input dominance data
dominance_edges <- data.frame(donor = substr(as.character(dominance_data$dyad), 1, 3), recipient = substr(as.character(dominance_data$dyad), 5, 7), dominance = dominance_data$dominance, squad = dominance_data$squad) # create network edge data frame by splitting "dyad" into donor and recipient

dominance_data1 <- subset(dominance_edges, squad == 1)      # subset out squad 1 data
dominance_data1b <- data.frame(donor = dominance_data1$recipient, recipient = dominance_data1$donor, dominance = abs(1 - dominance_data1$dominance), squad = dominance_data1$squad)  # create data frame that reverses donor/recipient and dominance
dominance_data1 <- rbind(dominance_data1, dominance_data1b) # combine both data frames

dominance_data2 <- subset(dominance_edges, squad == 2)      # subset out squad 2 data
dominance_data2b <- data.frame(donor = dominance_data2$recipient, recipient = dominance_data2$donor, dominance = abs(1 - dominance_data2$dominance), squad = dominance_data2$squad) # create data frame that reverses donor/recipient and dominance
dominance_data2 <- rbind(dominance_data2, dominance_data2b) # combine both data frames

##########################
## Functions
##########################
## Calculates rowwise variance
##  Input sharing matrix
rowwise_variance <- function(matrix) {
  row_var <- rep(NA, nrow(matrix))                  # initiate vector with a length of the number of rows in the matrix
  for(i in 1:nrow(matrix)) {                        # for each row in the matrix
    row_var[i] <- var(matrix[i, -i], na.rm = TRUE)  # calculate the variance of the matrix row
  }
  row_var   # output the vector of rowwise variances
}

## Calculates rowwise Spearman correlation (every row with it's transpose)
##  Input sharing matrix
spearman_rowwise_correlation <- function(matrix) {
  row_corr <- rep(NA, nrow(matrix)) # initiate vector with a length of the number of rows in the matrix
  for(i in 1:nrow(matrix)) {        # for each row in the matrix
    row_corr[i] <- cor(matrix[, i], matrix[i, ], method = "spearman", use = "complete.obs")# calculate the Spearman correlation of the matrix row
  }
  row_corr  # output the vector of rowwise Spearman correlations
}

## Generates random sharing matrices and calculates rowwise variance and Spearman correlation
##  Input sharing row sums, number of subjects, vectorized dominance adjacency matrix
rowwise_permutations <- function(sharing_row_sums, num_subjects, dominance_vector) {
  # Create random sharing edgelist
  random_share_edgelist_subject <- list(0)  # create an empty list
  for(j in 1:num_subjects){                 # for each subject
    recipient_nums <- (1:num_subjects)[-j]  # create vector of other subjects; num_subjects1 = rows in matrix; -j defines the set recipients you are interested in, i.e., everyone but the donor
    random_share_distribution <- sample(recipient_nums, sharing_row_sums[j], replace = TRUE) # for a given donor, we draw recipients at random, and we repeat this as many times as the row totals, i.e., we start with the number of times the donor was observed to share, then repeatedly sample form the other individuals, thus replace = T because the donor is allowed to choose the same recipient how ever many times.
    random_share_edgelist_subject[[j]] <- matrix(c(rep(j, length(random_share_distribution)), random_share_distribution), ncol = 2) # save the result for donor j as a matrix, where the first column is just the donor ID, repeated the number of times it shared, and the second column is the receiver of all those random shares
  }
  random_share_edgelist <- do.call(rbind, random_share_edgelist_subject) # the result, random_share_matrix_subject, is a list consisting of several two-column matrices. This function binds these matrices into one big two-column matrix, which is now our edgelist
  
  # Create random sharing matrix
  random_share_matrix <- matrix(NA, nrow = num_subjects, ncol = num_subjects) # initiate matrix
  for (rows in 1:num_subjects) { 		# for each row
    for (cols in 1:num_subjects) { 	# for each column
      random_share_matrix[rows, cols] <- sum(random_share_edgelist[, 1] == rows & random_share_edgelist[, 2] == cols) # sum the number of shares for the row and column dyad
    } # end cols
  } 	# end rows
  diag(random_share_matrix) <- NA # fill the matrix diagonal with NAs
  
  # Calculate rowwise variance and Spearman correlation
  random_rowwise_variance <- rowwise_variance(random_share_matrix)              # calculate rowwise variance
  random_rowwise_variance_sum <<- sum(random_rowwise_variance)                  # sum the variances
  random_share_matrix <- jitter(random_share_matrix, 0.01)                      # add jitter to aid in avoiding Spearman ties
  random_share_correlation <- spearman_rowwise_correlation(random_share_matrix) # calculate rowwise Spearman correlation
  random_share_correlation_mean <<- mean(random_share_correlation)              # calculate mean Spearman correlations
}

## Generates random sharing matrices and correlates each with empirical dominance matrix
##  Input sharing matrix graph to be randomized and dominance vector to be correlated
correlate_sharing_dominance <- function(sharing_matrix, dominance_vector) {
  # Recode any ties (0.5) to 1 for the top half of the matrix
  for(rows in 1:nrow(sharing_matrix)) { # for each row
    for(cols in 1:ncol(sharing_matrix)) { # for each column
      if(cols > rows & sharing_matrix[rows, cols] == 0.5) sharing_matrix[rows, cols] <- 1   # if in the top half of the matrix and the value = 0.5, replace the value with 1
    }
  }
  
  # Create random sharing matrix by maintaining matrix structure and shuffling subject IDs
  sharing_matrix_names <- rownames(sharing_matrix) # extract row names (subject IDs)
  random_names <- sample(sharing_matrix_names, length(sharing_matrix_names), replace = FALSE)   # randomly shuffle names
  random_sharing_matrix <- sharing_matrix                                                       # copy matrix
  rownames(random_sharing_matrix) <- colnames(random_sharing_matrix) <- random_names            # rename rows and columns to random_names
  random_sharing_matrix <- random_sharing_matrix[order(random_names), order(random_names)]      # reorder rows and columns based on names
  
  # Correlate sharing matrix with empirical dominance matrix
  random_sharing_vector <- as.vector(random_sharing_matrix[upper.tri(random_sharing_matrix)])   # convert upper triangle of random sharing matrix to vector
  random_share_dominance_correlation <- as.numeric(cor.test(random_sharing_vector, dominance_vector, na.rm = TRUE, method = "pearson")[4])  # calculate correlation between sharing vector and dominance vector
  random_share_dominance_correlation    # output correlation
}

##########################
## Plot sharing distributions
##########################
# Prepare data
sharing_trimmed <- subset(sharing_data, shares > 0)                                       # subset only sessions with sharing occurring
sharing_trimmed$squad <- factor(sharing_trimmed$squad, labels = c("Squad 1", "Squad 2"))  # create column of squads
sharing_trimmed$donor <- as.factor(sharing_trimmed$donor)                                 # convert donor to factor

# Plot boxplots of shares per donor
sharing_trimmed1_plot <- xyplot(jitter(shares) ~ donor | squad, subset(sharing_trimmed, squad == "Squad 1"), horizontal = F)  # plot sharing boxplot for each donor in squad 1
sharing_trimmed2_plot <- xyplot(jitter(shares) ~ donor | squad, subset(sharing_trimmed, squad == "Squad 2"), horizontal = F)  # plot sharing boxplot for each donor in squad 2
sharing_trimmed_plot <- c(sharing_trimmed1_plot, sharing_trimmed2_plot)                                               				# concatenate plots
sharing_trimmed_plot <- update(sharing_trimmed_plot,
  ylab = "Number of food items shared", xlab = "Subject", scales=list(y = "same"),
  par.settings = list(axis.text = list(cex = 1.75), par.ylab.text = list(cex = 2.5), par.xlab.text = list(cex = 2.5), layout.heights = list(strip = 2)),
  strip = strip.custom(par.strip.text = list(cex = 2)),
  panel = function(x, y, ...) {
    mean_values <<- tapply(y, x, mean, na.rm=T)      	          	# calculate means
    median_values <<- tapply(y, x, median, na.rm=T)      	        # calculate medians
    panel.xyplot(x, y, cex = 1.25, ...)                						# plot session-wise data points
    panel.points(mean_values, pch = 18, cex = 2, col = "black")		# plot means as diamonds
    panel.points(median_values, pch = 3, cex = 2, col = "black")	# plot medians as crosses
  }
)
png(file = "figures/sharing_boxplot.png", height = 600, width = 1000) # open PNG device
plot(sharing_trimmed_plot)  # plot boxplot
dev.off() # close PNG device

##########################
## Correlate sharing with dominance
##########################
##################
## Squad 1
##################
# Prepare data
sharing_edges1 <- sharing_data1[, c("donor","recipient","shares")]                                            # create an edgelist, with the third column being shares
sharing_weighted_edges1 <- aggregate(shares ~ donor + recipient, data = sharing_edges1, sum)                  # sum the sharing data for each donor-recipient combo (both directions)
sharing_weighted_edges1 <- sharing_weighted_edges1[with(sharing_weighted_edges1, order(donor, recipient)), ]  # reorder by donor and recipient
sharing_graph1 <- graph.data.frame(sharing_weighted_edges1, directed = TRUE)                                  # create a weighted, directed network, with edge weights as raw # of shares.
E(sharing_graph1)$weight <- E(sharing_graph1)$shares                                                          # calculate weights
sharing_matrix1 <- as.matrix(get.adjacency(sharing_graph1, attr = "weight"))                                  # get the adjacency matrix, with # of shares as the cell values
diag(sharing_matrix1) <- NA                                                                                   # fill the matrix diagonal with NAs
num_subjects1 <- nrow(sharing_matrix1)                                                                        # get number of subjects from number of rows in sharing matrix
sharing_row_sums1 <- rowSums(sharing_matrix1, na.rm = TRUE)                                                   # calculate row sums for each bird to keep this constrained per permutation

# Calculate empirical values
empirical_rowwise_sharing_correlation1 <- spearman_rowwise_correlation(sharing_matrix1) # calculate rowwise (dyad-wise) Spearman correlation of empirical sharing matrix
empirical_sharing_correlation1 <- mean(empirical_rowwise_sharing_correlation1)          # calculate mean correlation over all donors
empirical_rowwise_variance1 <- rowwise_variance(sharing_matrix1)                        # calculate rowwise variance of empirical sharing matrix
empirical_sharing_variance1 <- sum(empirical_rowwise_variance1)                         # sum the variance over all donors

# Create dominance matrix
dominance_weighted_edges1 <- aggregate(dominance ~ donor + recipient, data = dominance_data1, mean)                 # sum up the data for each donor-recipient combo, so now this is a weighted edgelist with only one entry for each donor-recipient combo.
dominance_weighted_edges1 <- dominance_weighted_edges1[with(dominance_weighted_edges1, order(donor, recipient)), ]  # reorder by donor and recipient
dominance_graph1 <- graph.data.frame(dominance_weighted_edges1)                                                     # create weighted, directed network, with edge weights as raw # of shares
E(dominance_graph1)$weight <- E(dominance_graph1)$dominance                                                         # calculate weights
dominance_matrix1 <- as.matrix(get.adjacency(dominance_graph1, attr = "weight"))                                    # get the adjacency matrix, with # of shares as the cell values
dominance_matrix1 <- as.matrix(dominance_matrix1)                                                                   # matrix of total # of shares
diag(dominance_matrix1) <- NA                                                                                       # fill the matrix diagonal with NAs

# Binarize matrices
dominance_data1_01 <- ifelse(dominance_matrix1 > 0.5, 1, ifelse(dominance_matrix1 == 0.5, 0.5, 0))                                        # convert to binary dominance matrix (1 = more dominant)
sharing_matrix1_01 <- ifelse((sharing_matrix1 - t(sharing_matrix1)) < 0, 0, ifelse((sharing_matrix1 - t(sharing_matrix1)) == 0, 0.5, 1))  # convert to binary sharing matrix (1 = shares more) and make ties 0.5

# Vectorize matrices
dominance_vector1 <- dominance_data1_01[upper.tri(dominance_data1_01)]  # convert upper triangle of dominance matrix to vector
sharing_vector1 <- sharing_matrix1_01[upper.tri(sharing_matrix1_01)]    # convert upper triangle of empirical sharing matrix to vector
empirical_dominance_sharing_corr1 <- as.numeric(cor.test(sharing_vector1, dominance_vector1, na.rm = TRUE, method = "pearson", na.rm = TRUE)[4])  # calculate correlation between sharing vector and dominance vector

##################
## Squad 2
##################
# Prepare data
sharing_edges2 <- sharing_data2[, c("donor","recipient","shares")]                                              # create an edgelist, with the third column being shares
sharing_weighted_edges2 <- aggregate(shares ~ donor + recipient, data = sharing_edges2, sum)                    # sum the sharing data for each donor-recipient combo (both directions)
sharing_weighted_edges2 <- sharing_weighted_edges2[with(sharing_weighted_edges2, order(donor, recipient)), ]    # reorder by donor and recipient
sharing_graph2 <- graph.data.frame(sharing_weighted_edges2, directed = TRUE)                                    # create a weighted, directed network, with edge weights as raw # of shares.
E(sharing_graph2)$weight <- E(sharing_graph2)$shares                                                            # calculate weights
sharing_matrix2 <- as.matrix(get.adjacency(sharing_graph2, attr = "weight"))                                    # get the adjacency matrix, with # of shares as the cell values
diag(sharing_matrix2) <- NA                                                                                     # fill the matrix diagonal with NAs
num_subjects2 <- nrow(sharing_matrix2)                                                                          # get number of subjects from number of rows in sharing matrix
sharing_row_sums2 <- rowSums(sharing_matrix2, na.rm = TRUE)                                                     # calculate row sums for each bird to keep this constrained per permutation

# Calculate empirical values
empirical_rowwise_sharing_correlation2 <- spearman_rowwise_correlation(sharing_matrix2)   # calculate rowwise (dyad-wise) Spearman correlation of empirical sharing matrix
empirical_sharing_correlation2 <- mean(empirical_rowwise_sharing_correlation2)            # calculate mean correlation over all donors
empirical_rowwise_variance2 <- rowwise_variance(sharing_matrix2)                          # calculate rowwise variance of empirical sharing matrix
empirical_sharing_variance2 <- sum(empirical_rowwise_variance2)                           # sum the variance over all donors

# Create dominance matrix
dominance_weighted_edges2 <- aggregate(dominance ~ donor + recipient, data = dominance_data2, mean)                 # sum up the data for each donor-recipient combo, so now this is a weighted edgelist with only one entry for each donor-recipient combo.
dominance_weighted_edges2 <- dominance_weighted_edges2[with(dominance_weighted_edges2, order(donor, recipient)), ]  # reorder by donor and recipient
dominance_graph2 <- graph.data.frame(dominance_weighted_edges2)                                                     # create weighted, directed network, with edge weights as raw # of shares
E(dominance_graph2)$weight <- E(dominance_graph2)$dominance                                                         # calculate weights
dominance_matrix2 <- as.matrix(get.adjacency(dominance_graph2, attr = "weight"))                                    # get the adjacency matrix, with # of shares as the cell values
dominance_matrix2 <- as.matrix(dominance_matrix2)                                                                   # matrix of total # of shares
diag(dominance_matrix2) <- NA                                                                                       # fill the matrix diagonal with NAs

# Binarize matrices
dominance_data2_01 <- ifelse(dominance_matrix2 > 0.5, 1, ifelse(dominance_matrix2 == 0.5, 0.5, 0))                                        # convert to binary dominance matrix (1 = more dominant) and make ties 0.5
sharing_matrix2_01 <- ifelse((sharing_matrix2 - t(sharing_matrix2)) < 0, 0, ifelse((sharing_matrix2 - t(sharing_matrix2)) == 0, 0.5, 1))  # convert to binary sharing matrix (1 = shares more) and make ties 0.5

# Vectorize matrices
dominance_vector2 <- dominance_data2_01[upper.tri(dominance_data2_01)]  # convert upper triangle of dominance matrix to vector
dominance_vector2[1] <- abs(1 - dominance_vector2[1])                   # switch first element from 1 to 0 to avoid standard deviation of 0
sharing_vector2 <- sharing_matrix2_01[upper.tri(sharing_matrix2_01)]    # convert upper triangle of empirical sharing matrix to vector
sharing_vector2[1] <- abs(1 - sharing_vector2[1])                       # switch first element from 1 to 0 to avoid standard deviation of 0
empirical_dominance_sharing_corr2 <- as.numeric(cor.test(sharing_vector2, dominance_vector2, na.rm = TRUE, method = "pearson", na.rm = TRUE)[4])  # calculate correlation between sharing vector and dominance vector

###########################
## Permutations
############################
iterations <- 10000 # set number of iterations for permutations

# Initiate empty vectors for sharing correlations, rowwise variance, and dominance/sharing correlations
random_share_correlation_mean1 <- random_share_correlation_mean2 <-
  random_rowwise_variance_sum1 <- random_rowwise_variance_sum2 <-
  random_share_dominance_correlation1 <- random_share_dominance_correlation2 <- rep(NA, iterations)

for (i in 1:iterations){  # for each permuation
  # Run rowwise permutations
  rowwise_permutations(sharing_row_sums1, num_subjects1)  # run rowwise permutations for squad 1
  rowwise_permutations(sharing_row_sums2, num_subjects2)  # run rowwise permutations for squad 2
  
  # Assign rowwise variance
  random_rowwise_variance_sum1[i] <- random_rowwise_variance_sum  # squad 1
  random_rowwise_variance_sum2[i] <- random_rowwise_variance_sum  # squad 2
  
  # Assign sharing correlation
  random_share_correlation_mean1[i] <- random_share_correlation_mean  # squad 1
  random_share_correlation_mean2[i] <- random_share_correlation_mean  # squad 2
  
  # Generates random dominance matrices and correlates each with empirical sharing matrix
  random_share_dominance_correlation1[i] <- correlate_sharing_dominance(sharing_matrix1, dominance_vector1)   # squad 1
  random_share_dominance_correlation2[i] <- correlate_sharing_dominance(sharing_matrix2, dominance_vector2)   # squad 2
}

##################
## Variance in sharing
##################
# Calculate median value of rowwise variances for each squad
random_rowwise_variance_median1 <- median(random_rowwise_variance_sum1)   # squad 1
random_rowwise_variance_median2 <- median(random_rowwise_variance_sum2)   # squad 2

# Calculate Fisher's exact test as proportion of distribution less than or equal to empirical value
p_value_rowwise_variance1 <- (length(which(random_rowwise_variance_sum1 >= empirical_sharing_variance1)) + 1) / (length(random_rowwise_variance_sum1) + 1)  # squad 1
p_value_rowwise_variance2 <- (length(which(random_rowwise_variance_sum2 >= empirical_sharing_variance2)) + 1) / (length(random_rowwise_variance_sum2) + 1)  # squad 2

# Combine squad data into a data frame
random_rowwise_variance_sum <- c(random_rowwise_variance_sum1, random_rowwise_variance_sum2)            # create vector of variances for both squads
squad <- c(rep(1, length(random_rowwise_variance_sum1)), rep(2, length(random_rowwise_variance_sum2)))  # create vector of squad numbers
rowwise_variance_df <- data.frame(variance = random_rowwise_variance_sum, squad = factor(squad))        # create data frame of variances and squads

# Plot histogram of random distribution and empirical values for each squad
rowwise_variance_hist <- histogram(~ variance | factor(squad, labels = c("Squad 1", "Squad 2")), data = rowwise_variance_df,
  layout = c(2, 1), breaks = NULL, col = "#C1CDCD",
  xlab = "Sharing variance", xlim = c(-70, 2500),
  par.settings = list(axis.text = list(cex = 1.75), par.ylab.text = list(cex = 2.5), par.xlab.text = list(cex = 2.5), layout.heights = list(strip = 2)),
  strip = strip.custom(par.strip.text = list(cex = 2)),
  panel = function(x, ...) {
    panel.histogram(x, ...)
    panel.number = panel.number()
    if (panel.number == 1) { 																												# for squad 1 panel
      panel.abline(v = empirical_sharing_variance1, col = "red", lty = 2, lwd = 2)  # plot empirical line
      panel.abline(v = random_rowwise_variance_median1, col = "blue", lwd = 1)      # plot random median line
    }
    if (panel.number == 2) { 																												# for squad 2 panel
      panel.abline(v = empirical_sharing_variance2, col = "red", lty = 2, lwd = 2)  # plot empirical line
      panel.abline(v = random_rowwise_variance_median2, col = "blue", lwd = 1)      # plot random median line
    }
  }
)
png(file = "figures/rowwise_variance.png", height = 600, width = 1000)  # open PNG device
plot(rowwise_variance_hist) # plot histogram
dev.off() # close PNG device

##################
## Sharing correlation
##################
# Calculate median value of sharing correlations for each squad
random_share_correlation_median1 <- median(random_share_correlation_mean1)  # squad 1
random_share_correlation_median2 <- median(random_share_correlation_mean2)  # squad 2

# Calculate Fisher's exact test as proportion of distribution less than or equal to empirical value
p_value_share_correlation1 <- (length(which(random_share_correlation_mean1 >= empirical_sharing_correlation1)) + 1) / (length(random_share_correlation_mean1) + 1)  # squad 1
p_value_share_correlation2 <- (length(which(random_share_correlation_mean2 >= empirical_sharing_correlation2)) + 1) / (length(random_share_correlation_mean2) + 1)  # squad 2

# Combine squad data into a data frame
random_share_correlation <- c(random_share_correlation_mean1, random_share_correlation_mean2)               # create vector of correlations for both squads
squad <- c(rep(1, length(random_share_correlation_mean1)), rep(2, length(random_share_correlation_mean2)))  # create vector of squad numbers
share_correlation_df <- data.frame(correlation = random_share_correlation, squad = factor(squad))           # create data frame of variances and squads

# Plot historgram of random distribution and empirical values for each squad
share_correlation_hist <- histogram(~ correlation | factor(squad, labels = c("Squad 1", "Squad 2")), data = share_correlation_df,
  layout = c(2, 1), breaks = NULL, col = "#C1CDCD",
  xlab = "Sharing given/received correlation", xlim = c(-1.1, 1.1),
  par.settings = list(axis.text = list(cex = 1.75), par.ylab.text = list(cex = 2.5), par.xlab.text = list(cex = 2.5), layout.heights = list(strip = 2)),
  strip = strip.custom(par.strip.text = list(cex = 2)),
  panel = function(x, ...) {
    panel.histogram(x, ...)
    panel.number = panel.number()
    if (panel.number == 1) { 																														# for squad 1 panel
      panel.abline(v = empirical_sharing_correlation1, col = "red", lty = 2, lwd = 2)  	# plot empirical line
      panel.abline(v = random_share_correlation_median1, col = "blue", lwd = 1)        	# plot random median line
    }
    if (panel.number == 2) { 																														# for squad 2 panel
      panel.abline(v = empirical_sharing_correlation2, col = "red", lty = 2, lwd = 2)  	# plot empirical line
      panel.abline(v = random_share_correlation_median2, col = "blue", lwd = 1)        	# plot random median line
    }
  }
)
png(file = "figures/share_correlation.png", height = 600, width = 1000)  # open PNG device
plot(share_correlation_hist)  # plot histogram
dev.off()  # close PNG device

##################
## Dominance and sharing correlation
##################
# Calculate median value of rowwise variances for each squad
random_share_dominance_median1 <- median(random_share_dominance_correlation1)   # squad 1
random_share_dominance_median2 <- median(random_share_dominance_correlation2)   # squad 2

# To test whether the empirical sharing/dominance correlation differs from the random distribution, we use a Fisher's exact test to calculate whether the empirical values falls in the outer tails of the random distribution.  Because the empirical value can fall in either tail, we tested the value against the nearest tail and set alpha = 0.025 to account for testing both directions.
p_value_dominance_sharing1 <- (length(which(random_share_dominance_correlation1 <= empirical_dominance_sharing_corr1)) + 1) / (length(random_share_dominance_correlation1) + 1)  # squad 1
p_value_dominance_sharing2 <- (length(which(random_share_dominance_correlation2 >= empirical_dominance_sharing_corr2)) + 1) / (length(random_share_dominance_correlation2) + 1)  # squad 2

# Combine squad data into a data frame
random_share_dominance_correlation <- c(random_share_dominance_correlation1, random_share_dominance_correlation2)     # create vector of correlations for both squads
squad <- c(rep(1, length(random_share_dominance_correlation1)), rep(2, length(random_share_dominance_correlation2)))  # create vector of squad numbers
share_dominance_df <- data.frame(correlation = random_share_dominance_correlation, squad = factor(squad))             # create data frame of correlations and squads

# Plot histogram of random distribution and empirical values for each squad
share_dominance_hist <- histogram(~ correlation | factor(squad, labels = c("Squad 1", "Squad 2")), data = share_dominance_df,
  layout = c(2, 1), breaks = 10, col = "#C1CDCD",
  xlab = "Dominance/sharing correlation coefficient",
  par.settings = list(axis.text = list(cex = 1.75), par.ylab.text = list(cex = 2.5), par.xlab.text = list(cex = 2.5), layout.heights = list(strip = 2)),
  strip = strip.custom(par.strip.text = list(cex = 2)),
  panel = function(x, ...) {
    panel.histogram(x, ...)
    panel.number = panel.number()
    if (panel.number == 1) { 																															# for squad 1 panel
      panel.abline(v = empirical_dominance_sharing_corr1, col = "red", lty = 2, lwd = 2)  # plot empirical line
      panel.abline(v = random_share_dominance_median1, col = "blue", lwd = 1)             # plot random median line
    }
    if (panel.number == 2) { 																															# for squad 2 panel
      panel.abline(v = empirical_dominance_sharing_corr2, col = "red", lty = 2, lwd = 2)  # plot empirical line
      panel.abline(v = random_share_dominance_median2, col = "blue", lwd = 1)             # plot random median line
    }
  }
)
png(file = "figures/share_dominance.png", height = 600, width = 1000)  # open PNG device
plot(share_dominance_hist) # plot histogram
dev.off()  # close PNG device

##########################
## Conduct GLMMs
##########################
## Squad 1
# Calculate null GLMM model
null_glm1 <- glmer(any_shares ~ (1 | donor) + (1 | dyad), family = "binomial", data = sharing_data1)
summary(null_glm1)

# Calculate direct reciprocity GLMM model
direct_glm1 <- glmer(any_shares ~ direct_recip + (1 | donor) + (1 | dyad), family = "binomial", data = sharing_data1)
summary(direct_glm1)

# Calculate generalized reciprocity GLMM model
general_glm1 <- glmer(any_shares ~ general_recip + (1 | donor) + (1 | dyad), family = "binomial", data = sharing_data1)
summary(general_glm1)

## Squad 2
# Calculate null GLMM model
null_glm2 <- glmer(any_shares ~ (1 | donor) + (1 | dyad), family = "binomial", data = sharing_data2)
summary(null_glm2)

# Calculate direct reciprocity GLMM model
direct_glm2 <- glmer(any_shares ~ direct_recip + (1 | donor) + (1 | dyad), family = "binomial", data = sharing_data2)
summary(direct_glm2)

# Calculate generalized reciprocity GLMM model
general_glm2 <- glmer(any_shares ~ general_recip + (1 | donor) + (1 | dyad), family = "binomial", data = sharing_data2)
summary(general_glm2)

##########################
# Calculate individual direct and generalized reciprocity rates
##########################

# Direct reciprocity
after_direct <- subset(sharing_data, direct_recip == 1)	# subset the cases in which that partner shared last time
direct_sharing_subject <- aggregate(any_shares ~ donor * squad, data = after_direct, FUN = mean) 	# calculate the mean number of sharing events for each donor
names(direct_sharing_subject)[3] <- "after_share"		# rename column
direct_sharing_subject$N_after_share <- aggregate(any_shares ~ donor * squad, data = after_direct, FUN = length)$any_shares # calculate the number of cases in which a donor's partner shared previously
after_no_direct <- subset(sharing_data, direct_recip == 0)    # subset the cases in which that partner DID NOT SHARE previously
direct_sharing_subject$after_no_share <- aggregate(any_shares ~ donor * squad, data = after_no_direct, FUN = mean)$any_shares	# calculate the mean number of sharing events for each donor (share following a non-share)
direct_sharing_subject$N_after_no_share <- aggregate(any_shares ~ donor * squad, data = after_no_direct, FUN = length)$any_shares  # calculate the number of cases in which a donor's partner did NOT share previously  
direct_sharing_subject$reciprocity <- direct_sharing_subject$after_share - direct_sharing_subject$after_no_share	# calculate difference

# Generalized reciprocity
after_general <- subset(sharing_data, general_recip == 1)		# subset the cases in which that partner shared last time
general_sharing_subject <- aggregate(any_shares ~ donor * squad, data = after_general, FUN = mean) # calculate the mean number of sharing events for each donor
names(general_sharing_subject)[3] <- "after_share"		# rename column
general_sharing_subject$N_after_share <- aggregate(any_shares ~ donor * squad, data = after_general, FUN = length)$any_shares # calculate the number of cases in which a donor's partner shared previously
after_no_general <- subset(sharing_data, general_recip == 0)  # subset the cases in which that partner DID NOT SHARE previously  
general_sharing_subject$after_no_share <- aggregate(any_shares ~ donor * squad, data = after_no_general, FUN = mean)$any_shares	# calculate the mean number of sharing events for each donor (share following a non-share)
general_sharing_subject$N_after_no_share <- aggregate(any_shares ~ donor * squad, data = after_no_general, FUN = length)$any_shares    # calculate the number of cases in which a donor's partner did NOT share previously
general_sharing_subject$reciprocity <- general_sharing_subject$after_share - general_sharing_subject$after_no_share	# calculate difference

# Combine data
sharing_subjects <- rbind(direct_sharing_subject, general_sharing_subject)	# combine direct and generalized reciprocity data
sharing_subjects$type <- as.factor(rep(c("Direct reciprocity", "Generalized reciprocity"), each = 11))	# label type of reciprocity

# Plot figure
reciprocity_plot <- xyplot(reciprocity ~ factor(squad, labels = c("Squad 1", "Squad 2")) | type, data = sharing_subjects, 
  cex = 2, pch = c(1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2), col = c(rep(col.blind[1], 6), rep(col.blind[2], 5)), 
  xlab = NULL, ylab = "Proportion sharing \nmore following sharing", ylim = c(-1.1, 1.1),
  par.settings = list(axis.text = list(cex = 2), par.ylab.text = list(cex = 2.5), par.xlab.text = list(cex = 2.5), layout.heights = list(strip = 2)),
  strip = strip.custom(par.strip.text = list(cex = 2)),
  scales = list(alternating = FALSE),
  panel = function(x, y, ...) {
    panel.xyplot(x, y, ...)
    panel.abline(h = 0, lty = 2)
  }
)
png(file = "figures/reciprocity_plot.png", height = 600, width = 800)  # open PNG device
plot(reciprocity_plot) # plot histogram
dev.off()  # close PNG device
