% This is the primary script for running the evolutionary simulations
% To run this simulation, put all scripts in a single directory, and
% create a directory named "data" (if not already existent). 

%%%%%%%%%%%%%%%%%%%%%%%
% Clear and assign variables
%%%%%%%%%%%%%%%%%%%%%%%
clear;  % clear variables
clearvars;  % clear variables
numIndivs=90;   % number of individuals
numGenerations= 250;     % number of generations
memoryParam1=-1;    % lambda for memory function         (-1: no application of memory function/perfect recall)
memoryParam2=-1;    % psi for memory function	(decay rate)
numStrategies=9;    % number of strategies
numStructures=3;    % number of contact patterns (skew)
numMemories=7;      % number of forgetting rates
numMemoryTypes=2;   % number of memory error types 1: random, 2: forget
numRepetitions=1000;  % number of simulation repetitions
imperfectOwnActions=1;  % own action can be forgotten (if =1)
numChosen = 20;     % number of highest fitness agents used for truncation selection operator
deterministic = 0;  % flag for deterministic reproduction (0 = reproduce equally) or stochastic (1 = reproduce with uniform probability 1/p) truncation selection operator
bAnalysisOnly=0;    % flag for test analysis with limited condition, repetitions, and generations and no data recording (0=no; 1=yes)
bDistractor=0;      % flag for including additional distractor interactions (0=no; 1=yes)
numDistractionsPerIndiv=100;    % number of distractor interactions
fileDir='./data/';  % directory name for output: comment when running on Microsoft Windows
% fileDir='.\data\';  % directory name for output: uncomment when running on Microsoft Windows
tic;    % start timer
if bAnalysisOnly    % when running test analysis, assign parameters
    numGenerations=1;
    numRepetitions=1;
    numMemoryTypes=1;
    numMemories=1;
end

%%%%%%%%%%%%%%%%%%%%%%%
% Primary control code
%%%%%%%%%%%%%%%%%%%%%%%
for MemoryT=1:numMemoryTypes    % for each memory error type
    for MemoryI=1:numMemories   % for each forgetting rate
        if or((MemoryT==1), (MemoryI>1))   
            for structuresI=1:numStructures     % for each contact pattern
                pairStrengthSequence=zeros(numIndivs-1,1);
                switch structuresI  % Contact patterns
                    case 1  % High skew
                        pairStrengthSequence(1:10,1)=[33 23 15 10 7 5 3 2 1 1];                               % number of interactions with the 10 interaction partners (high skew)
                        structName='VerySkewed';
                        numContacts=10;
                    case 2  % Low skew
                        pairStrengthSequence(1:10,1)=[19 16 14 11 10 8 7 6 5 4];								% number of interactions with the10 interaction partners (low skew)
                        structName='LessSkewed';
                        numContacts=10;     
                    case 3  % No skew
                        pairStrengthSequence(1:10,1)=10;																		% 10 interactions with each of the 10 interaction partners (no skew)
                        structName='Equal';
                        numContacts=10;
                end
                switch MemoryI  % Probability of corrrect recall: p = λ * ((k + 1) ^ (− ψ)) 
											  %		 λ (lambda) starting point of the forgetting function 
											  % 		ψ (psi) decay rate of the forgetting function 
											  % 		k number of interactions between the target and the current interaction (k = 0 if the target interaction was the most recent interaction)
                    case 1						% no forgetting
                        Lambda=1;
                        Psi=0;
                          memName='L1P00';
                    case 2
                        Lambda=1;
                        Psi=0.10;
                          memName='L1P10';
                    case 3
                        Lambda=1;
                        Psi=0.25;
                        memName='L1P25';
                    case 4
                        Lambda=0.25;
                        Psi=0.50;
                        memName='L25P50';  
                    case 5  % Empirical forgetting rate
                        Lambda=0.8775247;
                        Psi=0.2289332;
                        memName='L87P22';      
                    case 6
                        Lambda=0.4;
                        Psi=0.25;
                        memName='L40P25';       
                    case 7
                        Lambda=0.6;
                        Psi=0.25;
                        memName='L60P25';       
                end
                switch MemoryT  % Memory error types
                    case 1  % Errors of commission
                        MemoryType='Random';
                        memTName='RND';
                    case 2  % Errors of omission
                        MemoryType='Forget';
                        memTName='FGT';
                end        
                typeProportions(1:numStrategies,1)=1/numStrategies;                   %proportion of agents playing each type of strategy in the starting population
                numTypes=size(typeProportions,1);													%number of different strategies still played in the population			 
                matrixSave=zeros(numRepetitions, numGenerations, numStrategies);				
                matrixOutcomeSave=zeros(numRepetitions, numGenerations, numStrategies);
                matrixInteractionSave=zeros(numRepetitions, numGenerations, numStrategies,numStrategies);                  
                matrixInteractionMeanSave=zeros(numRepetitions, numGenerations, numStrategies,numStrategies);
                matrixInteractionCountSave=zeros(numRepetitions, numGenerations, numStrategies,numStrategies);
                matrixCooperationRatioSave=zeros(numRepetitions, numGenerations);																% stores overall level of cooperation for all generations in all runs
                matrixCooperationRatioStratSave=zeros(numRepetitions, numGenerations,numStrategies);							% stores the ratio of strategies for all generations in all runs
                for ci = 1:numRepetitions                                                 % runs through all simulation repetitions
                    disp(strcat(int2str(ci),': ',num2str(toc)));													
                    TypeHist=zeros(numGenerations,numTypes);											% stores the proportion of agents playing each type of strategy for all generations this simulation run
                    OutcomeHist=zeros(numGenerations,numTypes);									% stores the score for each type of strategy for all generations this simulation run
                    InteractionHist=zeros(numGenerations,numTypes,numTypes);            % 
                    InteractionMeanHist=zeros(numGenerations,numTypes,numTypes);
                    CountHist=zeros(numGenerations,numTypes,numTypes);
                    CoopRatioHist=zeros(numGenerations,1);
                    CoopRatioHistStrat=zeros(numGenerations,numTypes);
                    playerTypes=  createPopulation(numIndivs,typeProportions);																			%contains the strategies (1-9) of all agents
                    pairConnections = createPairConnections( numIndivs, numContacts, pairStrengthSequence );               % creates a matrix specifying the number of interaction of each agent with each other agent  
                    for gi=1:numGenerations
                        disp(strcat('T',int2str(ci),'G',int2str(gi),': ',num2str(toc)));																				%outputs simulation number, generation number and time stamp
                        contactList=createContactListRandomOrderDistractor(pairConnections,bDistractor,numDistractionsPerIndiv);
                        if bAnalysisOnly																											% only for flag for test analysis with limited condition, repetitions, and generations and no data recording 
                            sumConnect=sum(pairStrengthSequence,1);
                            maxConnect=max(pairStrengthSequence,[],1);
                            [summary1, summary2]=analyzeContactList(contactList, numIndivs, numContacts,sumConnect,maxConnect);   % outputs matrix with interaction frequency of all agents with all other agents; and mean interaction number for each agent
                        else 
                            [indivOutcomes, typeMatrix, typeMeanMatrix, typeCountMatrix, cooperationRatio, cooperationRatioStrat]=playGames(contactList,playerTypes,Lambda,Psi,MemoryType,numStrategies,imperfectOwnActions);
                            a=cooperationRatioStrat(1,:);
                            if a<1
                                b=a;
                            end
                            % Selection operators (uncomment desired selection operator
                            [playerTypes, typeCount, typeOutcome]=generateNewPopulationRoulette(indivOutcomes, playerTypes,numTypes);
%                             [playerTypes, typeCount, typeOutcome]=generateNewPopulationSUS(indivOutcomes, playerTypes,numTypes);
%                             [playerTypes, typeCount, typeOutcome]=generateNewPopulationTruncation(indivOutcomes, playerTypes,numTypes,numChosen,deterministic);

                            TypeHist(gi,1:numTypes)=typeCount(1:numTypes,1)./numIndivs;
                            OutcomeHist(gi,1:numTypes)=typeOutcome(1:numTypes,1);
                            InteractionHist(gi,1:numTypes,1:numTypes)=typeMatrix(1:numTypes,1:numTypes);
                            InteractionMeanHist(gi,1:numTypes,1:numTypes)=typeMeanMatrix(1:numTypes,1:numTypes);
                            CountHist(gi,1:numTypes,1:numTypes)=typeCountMatrix(1:numTypes,1:numTypes);
                            CoopRatioHist(gi,1)=cooperationRatio;
                            CoopRatioHistStrat(gi,1:numTypes)=cooperationRatioStrat(1:numTypes);
                        end   
                    end
                    if ~bAnalysisOnly 
                        matrixSave(ci, 1:numGenerations, 1:numTypes)=TypeHist(1:numGenerations, 1:numTypes);
                        matrixOutcomeSave(ci, 1:numGenerations, 1:numTypes)=OutcomeHist(1:numGenerations, 1:numTypes);
                        matrixInteractionSave(ci,1:numGenerations,1:numTypes,1:numTypes)=InteractionHist(1:numGenerations, 1:numTypes, 1:numTypes);
                        matrixInteractionMeanSave(ci,1:numGenerations,1:numTypes,1:numTypes)=InteractionMeanHist(1:numGenerations, 1:numTypes, 1:numTypes);
                        matrixInteractionCountSave(ci,1:numGenerations,1:numTypes,1:numTypes)=CountHist(1:numGenerations, 1:numTypes, 1:numTypes);
                        matrixCooperationRatioSave(ci,1:numGenerations)=CoopRatioHist(1:numGenerations,1);
                        matrixCooperationRatioStratSave(ci,1:numGenerations,1:numTypes)=CoopRatioHistStrat(1:numGenerations,1:numTypes);
                    end
                end
                if ~bAnalysisOnly 
                    matrixInteractionSave=squeeze(nanmean(matrixInteractionSave,1));
                    matrixInteractionMeanSave=squeeze(nanmean(matrixInteractionMeanSave,1));
                    matrixInteractionCountSave=squeeze(nanmean(matrixInteractionCountSave,1));
                    partMatrix=zeros(numRepetitions, numGenerations, numTypes);
                    partMatrixMean=zeros(numStrategies,numGenerations);
                    partMatrixSurvival=zeros(numStrategies,numGenerations);
                    partMatrixDominance=zeros(numStrategies,numGenerations);
                    partOutcomeMean=squeeze(nanmean(matrixOutcomeSave,1));
                    meanTempRatioStratSave=squeeze(nanmean(matrixCooperationRatioStratSave,1));
                    forSummaryMatrix=zeros(numRepetitions,numTypes);
                    for ti=1:numTypes 
                        partMatrixInteraction=zeros(numGenerations,numStrategies);
                        partMatrixInteractionMean=zeros(numGenerations,numStrategies);
                        partMatrixCountMean=zeros(numGenerations,numStrategies);
                        fileName=strcat(fileDir,'SingleProportion','_M-',memName,'_E-',memTName,'_S-',structName,'_','T',int2str(ti),'.txt');
                        partMatrix=matrixSave(1:numRepetitions, 1:numGenerations, ti);
                        partMatrixMean(ti,1:numGenerations)=mean(partMatrix,1);
                        partMatrixSurvival(ti,1:numGenerations)=mean(partMatrix>0.00000000001,1);
                        partMatrixDominance(ti,1:numGenerations)=mean(partMatrix>0.9999999999,1);
                        dlmwrite(fileName, partMatrix, '\t');
                        forSummaryMatrix(1:numRepetitions,ti)=partMatrix(1:numRepetitions, numGenerations);
                        fileName=strcat(fileDir,'Interaction','_M-',memName,'_E-',memTName,'_S-',structName,'_','T',int2str(ti),'.txt');
                        partMatrixInteraction(1:numGenerations,1:numStrategies)=squeeze(matrixInteractionSave(1:numGenerations,ti,1:numStrategies));
                        dlmwrite(fileName, partMatrixInteraction, '\t');
                        fileName=strcat(fileDir,'InteractionMean','_M-',memName,'_E-',memTName,'_S-',structName,'_','T',int2str(ti),'.txt');
                        partMatrixInteractionMean(1:numGenerations,1:numStrategies)=squeeze(matrixInteractionMeanSave(1:numGenerations,ti,1:numStrategies));
                        dlmwrite(fileName, partMatrixInteractionMean, '\t');
                        fileName=strcat(fileDir,'InteractionCount','_M-',memName,'_E-',memTName,'_S-',structName,'_','T',int2str(ti),'.txt');
                        partMatrixCountMean(1:numGenerations,1:numStrategies)=squeeze(matrixInteractionCountSave(1:numGenerations,ti,1:numStrategies));
                        dlmwrite(fileName, partMatrixCountMean, '\t');  
                    end
                    tftSelection=[3; 4;  5;  8];
                    tftMatrix= forSummaryMatrix(:,tftSelection);
                    tftSumMatrix=sum(tftMatrix,2);  
                    fileName=strcat(fileDir,'EndStateProportions','_M-',memName,'_E-',memTName,'_S-',structName,'.txt');
                    dlmwrite(fileName, [forSummaryMatrix tftSumMatrix], '\t');  
                    shortSummaryMatrix=zeros(4,numTypes+1);
                    shortSummaryMatrix(1,1:numTypes)=mean(forSummaryMatrix,1);
                    shortSummaryMatrix(1,numTypes+1)=mean(tftSumMatrix,1);
                    shortSummaryMatrix(2,1:numTypes)=std(forSummaryMatrix,0,1);
                    shortSummaryMatrix(2,numTypes+1)=std(tftSumMatrix,0,1);
                    shortSummaryMatrix(3,1:numTypes)= meanTempRatioStratSave(numGenerations,1:numStrategies);
                    shortSummaryMatrix(3,numTypes+1)=    sum((shortSummaryMatrix(3,tftSelection).*shortSummaryMatrix(1,tftSelection)),2) / shortSummaryMatrix(1,numTypes+1);
                    shortSummaryMatrix(4,1)=mean(matrixCooperationRatioSave(:,numGenerations),1);
                    shortSummaryMatrix(4,2)=std(matrixCooperationRatioSave(:,numGenerations),0,1);
                    fileName=strcat(fileDir,'shortSummaryPaperXXXXX','_M-',memName,'_E-',memTName,'_S-',structName,'.txt');
                    dlmwrite(fileName, shortSummaryMatrix, '\t');
                    fileName=strcat(fileDir,'cooperationRatioStrat','_M-',memName,'_E-',memTName,'_S-',structName,'.txt');
                    partmatrixCooperationRatioStratSave(1:numGenerations,1:numStrategies)=meanTempRatioStratSave(1:numGenerations,1:numStrategies);
                    dlmwrite(fileName, partmatrixCooperationRatioStratSave, '\t');
                    fileName=strcat(fileDir,'cooperationRatio','_M-',memName,'_E-',memTName,'_S-',structName,'.txt');
                    dlmwrite(fileName, matrixCooperationRatioSave', '\t');
                    fileName=strcat(fileDir,'MeanProportion','_M-',memName,'_E-',memTName,'_S-',structName,'.txt');
                    dlmwrite(fileName, partMatrixMean', '\t');
                    fileName=strcat(fileDir,'MeanOutcome','_M-',memName,'_E-',memTName,'_S-',structName,'.txt');
                    dlmwrite(fileName, partOutcomeMean, '\t');
                    fileName=strcat(fileDir,'MeanSurvival','_M-',memName,'_E-',memTName,'_S-',structName,'.txt');
                    dlmwrite(fileName, partMatrixSurvival', '\t');
                    fileName=strcat(fileDir,'MeanDominance','_M-',memName,'_E-',memTName,'_S-',structName,'.txt');
                    dlmwrite(fileName, partMatrixDominance', '\t');
                    set(0,'DefaultAxesColorOrder',[1 0 0;0 1 0;0 0 1;0.5 0.5 0.5; 0 1 1; 1 1 0; 1 0 1],...
                      'DefaultAxesLineStyleOrder','-|--|:');
                    hFig = figure(1);% from figure
                    plot(partMatrixMean');
                    title('Type numbers per generation')
                    xlabel('Generation')
                    ylabel('Type Frequency')
                    legend('ALLC','ALLD','TFT','TF2T','GTFT','WSLS','RAND','CTFT','GRIM'); % Add a legend in the upper left:
                    Figname=strcat(fileDir,'fig_',memName,'_E-',memTName,'_',structName,'.png');
                    print(hFig, '-dpng', Figname);
                end
            end
        end
    end
end
disp(toc);  % display timer