## ---
##
## Script name: stevens_etal_2021_rcode.R
##
## Purpose of script: This script analyzes the data investigating the effect of unit framinig on similarity judgments and intertemporal choice
##
## Authors: Jeffrey R. Stevens (jeffrey.r.stevens@gmail.com) and Tyler Cully (tylercully@yahoo.com)
##
## Date Created: 2021-01-16
##
## Date Finalized: 2021-09-27
##
## License: All materials presented here are released under the Creative Commons Attribution 4.0 International Public License (CC BY 4.0).
##  You are free to:
##   Share — copy and redistribute the material in any medium or format
##   Adapt — remix, transform, and build upon the material for any purpose, even commercially.
##  Under the following terms:
##   Attribution — You must give appropriate credit, provide a link to the license, and indicate if changes were made. You may do so in any reasonable manner, but not in any way that suggests the licensor endorses you or your use.
##   No additional restrictions — You may not apply legal terms or technological measures that legally restrict others from doing anything the license permits.
##
## ---
##
## Data files:
## ---
## stevens_etal_data.csv
##  experiment - experiment name and number
##  date - date of experimental session
##  participant - participant ID
##  condition - experimental condition (dollars, cents, days, weeks)
##  type - type of trial/question
##  smallamount - small amount value
##  largeamount - large amount value
##  shortdelay - short delay value
##  longdelay - long delay value
##  response - response to question
##
## ---


# Libraries and functions -------------------------------------------------
library(BayesFactor)
library(bayestestR)
library(lme4)
library(performance)
library(mediation)
library(ggdist)
library(papaja)
library(tidyverse)


### Extract demographic information from data set
extract_demographics <- function(df, type = "unl") {
  age_data <- filter(df, type == "age") # filter age info
  age_mean <- mean(as.numeric(age_data$response), na.rm = TRUE) # calculate mean age
  age_min <- min(as.numeric(age_data$response), na.rm = TRUE) # calculate minimum age
  age_max <- max(as.numeric(age_data$response), na.rm = TRUE) # calculate maximum age
  
  sample_size <- nrow(age_data) # calculate sample size
  
  condition_table <- table(age_data$condition) # count number of participant per condition
  condition_ptable <- paste(round(condition_table / sum(condition_table) * 100, 1), "%", sep = "") # calculate percentage per condition
  names(condition_ptable) <- names(condition_table) # add table names
  
  gender_data <- filter(df, type == "gender") # filter gender info
  gender_table <- table(gender_data$response) # count number of participant per gender
  # If genders aren't represented, add them with 0 entries
  all_genders <- c("Male", "Female", "Other") # create vector of genders
  table_genders <- names(gender_table) # create vector of genders for which we have entries
  for (gender in all_genders) { # for each gender
    if (!gender %in% table_genders) { # check if there is no entry
      gender_missing <- 0 # assign 0
      names(gender_missing) <- gender # assign name
      gender_table <- c(gender_table, gender_missing) # append to table of genders
    }
  }
  gender_ptable <- paste(round(gender_table / sum(gender_table) * 100, 1), "%", sep = "") # calculate percentage per condition
  names(gender_ptable) <- names(gender_table) # add table names
  
  race_data <- filter(df, type == "race") %>%  # filter race info
    mutate(response = ifelse(str_detect(response, pattern = ", "), "Multiracial", response))
  race_table <- table(race_data$response) # count number of participant per race
  # If races aren't represented, add them with 0 entries
  if (type == "mturk") {
    all_races <- c("Black", "Asian", "NativeAmerican", "White", "Other", "Latino", "Multiracial") # create vector of races
  } else {
    all_races <- c("Black", "Asian", "NativeAmerican", "White", "Other")
  }
  table_races <- names(race_table) # create vector of races for which we have entries
  for (race in all_races) { # for each race
    if (!race %in% table_races) { # check if there is no entry
      race_missing <- 0 # assign 0
      names(race_missing) <- race # assign name
      race_table <- c(race_table, race_missing) # append to table of races
    }
  }
  race_ptable <- paste(round(race_table / sum(race_table) * 100, 1), "%", sep = "") # calculate percentage per condition
  names(race_ptable) <- names(race_table) # add table names
  
  income_data <- filter(df, type == "income") # filter income info
  income_table <- table(income_data$response) # count number of participant per income level
  all_incomes <- c("<$25000", "$25000", "$50000", "$75000", ">$100000", "Noanswer") # create vector of incomes
  table_incomes <- names(income_table) # create vector of incomes for which we have entries
  for (income in all_incomes) { # for each income
    if (!income %in% table_incomes) { # check if there is no entry
      income_missing <- 0 # assign 0
      names(income_missing) <- income # assign name
      income_table <- c(income_table, income_missing) # append to table of incomes
    }
  }
  income_ptable <- paste(round(income_table / sum(income_table) * 100, 1), "%", sep = "") # calculate percentage per condition
  names(income_ptable) <- names(income_table) # add table names
  
  demo_data <- list(sample_size = sample_size, condition = condition_table, condition_percent = condition_ptable, age_mean = age_mean, age_min = age_min, age_max = age_max, gender = gender_table, gender_percent = gender_ptable, race = race_table, race_percent = race_ptable, income = income_table, income_percent = income_ptable)
}

## Pivot data frames to have ITC, amount judgment, and delay judgments as columns
pivot_itc_wider <- function(df) {
  clean_data_wide <- df %>%
    unite("amt_pair", smallamount:largeamount, remove = FALSE) %>% # create column for amount pair
    unite("delay_pair", shortdelay:longdelay, remove = FALSE) %>% # create column for delay pair
    unite("question", amt_pair, delay_pair, remove = FALSE, sep = "-") # create column for question
  questions <- clean_data_wide %>%
    filter(type == "itc") %>%
    group_by(question) %>%
    slice(n = 1) %>% # extract single instance of each question
    pull(question) # create vector of questions
  amount_wide <- clean_data_wide %>%
    filter(type %in% c("itc", "amt_binary")) %>%
    pivot_wider(
      id_cols = c("participant", "condition", "amt_pair"), # pivot amt_binary with itc
      names_from = type, values_from = response, values_fn = list
    ) %>%
    rowwise() %>%
    mutate(question = str_subset(questions, amt_pair)) %>% # add full question column
    ungroup() %>%
    mutate(
      itc = as.character(itc), # convert list to character vector
      amt_binary = as.character(amt_binary)
    ) %>% # convert list to character vector
    separate(question, c("amt-pair", "delay_pair"), sep = "-", remove = FALSE)
  delay_wide <- clean_data_wide %>%
    filter(type %in% c("delay_binary")) %>%
    pivot_wider(
      id_cols = c("participant", "condition", "delay_pair"), # pivot delay_binary with itc
      names_from = type, values_from = response, values_fn = list
    ) # %
  all_wide <- amount_wide %>%
    left_join(delay_wide, by = c("participant", "condition", "delay_pair")) %>% # join delay to itc and amt
    select(participant, condition, question, amt_pair, delay_pair, itc, amt_binary, delay_binary) %>%
    mutate(
      itc = as.numeric(itc),
      delay_binary = as.character(delay_binary)
    ) # convert list to character vector
}

## Pivot data frames to have risk, amount judgment, and probability judgments as columns
pivot_risk_wider <- function(df) {
  clean_data_wide <- df %>%
    unite("amt_pair", smallamount:largeamount, remove = FALSE) %>% # create column for amount pair
    unite("prob_pair", smallprob:largeprob, remove = FALSE) %>% # create column for delay pair
    unite("question", amt_pair, prob_pair, remove = FALSE, sep = "-") # create column for question
  questions <- clean_data_wide %>%
    filter(type == "choice") %>%
    group_by(question) %>%
    slice(n = 1) %>% # extract single instance of each question
    pull(question) # create vector of questions
  amount_wide <- clean_data_wide %>%
    filter(type %in% c("choice", "amt_binary")) %>%
    pivot_wider(
      id_cols = c("participant", "condition", "amt_pair"), # pivot amt_binary with choice
      names_from = type, values_from = response, values_fn = list
    ) %>%
    rowwise() %>%
    mutate(question = str_subset(questions, paste(amt_pair, "-", sep = ""))) %>% # add full question column
    ungroup() %>%
    mutate(
      choice = as.character(choice), # convert list to character vector
      amt_binary = as.character(amt_binary)
    ) %>% # convert list to character vector
    separate(question, c("amt-pair", "prob_pair"), sep = "-", remove = FALSE)
  prob_wide <- clean_data_wide %>%
    filter(type %in% c("prob_binary")) %>%
    pivot_wider(
      id_cols = c("participant", "condition", "prob_pair"), # pivot prob_binary with choice
      names_from = type, values_from = response, values_fn = list
    ) # %
  all_wide <- amount_wide %>%
    left_join(prob_wide, by = c("participant", "condition", "prob_pair")) %>% # join prob to choice and amt
    select(participant, condition, question, amt_pair, prob_pair, choice, amt_binary, prob_binary) %>%
    mutate(
      choice = as.numeric(choice),
      prob_binary = as.character(prob_binary)
    ) # convert list to character vector
}

## Removes ITC participants not meeting eligibility criteria
process_itc_data <- function(df, type) {
  # Define bad participants
  bad_exclusions <- filter(df, type == "exclusions" & !is.na(response)) # find participants with exclusions flagged
  bad_ITC <- filter(df, type == "itc" & smallamount == largeamount & response == 1) # find participants who choose larger, later option when amounts are the same
  bad_amt <- filter(df, (type == "amt_binary" & smallamount == largeamount & response == 0) | (type == "amt_binary" & largeamount / smallamount >= 10 & response == 1)) # find participants who consider same amounts as dissimilar or who treat large ratios as similar
  bad_delay <- filter(df, (type == "delay_binary" & shortdelay == longdelay & response == 0) | (type == "delay_binary" & longdelay / shortdelay >= 10 & shortdelay != 0 & response == 1)) # find participants who consider same amounts as dissimilar or who treat large ratios as similar
  bad_participants <- bind_rows(bad_exclusions, bad_ITC, bad_amt, bad_delay)
  
  # Remove bad participants and attention check questions
  if (type == "amount") {
    clean_data <- df %>%
      anti_join(bad_participants, by = "participant") %>% # remove bad participants from clean data
      filter(is.na(smallamount) | smallamount != largeamount | type == "numberline") %>% # filter out rows with identical amounts
      filter(is.na(shortdelay) | shortdelay != longdelay) %>% # filter out rows with identical delays
      filter(is.na(smallamount) | largeamount / smallamount < 10 | type == "numberline") %>% # filter out rows with large amount ratios
      filter(is.na(shortdelay) | (longdelay / shortdelay < 10 | shortdelay == 0)) # filter out rows with large delay ratios
  } else if (type == "delay" | type == "delay_mturk") {
    clean_data <- df %>%
      anti_join(bad_participants, by = "participant") %>% # remove bad participants from clean data
      filter(is.na(smallamount) | smallamount != largeamount | type == "numberline") %>% # filter out rows with identical amounts
      filter(is.na(shortdelay) | shortdelay != longdelay) %>% # filter out rows with identical delays
      filter(is.na(smallamount) | largeamount / smallamount < 10 | type == "numberline") %>% # filter out rows with large amount ratios
      filter(is.na(shortdelay) | shortdelay != 0 | longdelay != 70) # filter out rows with large delay ratios
  } else {
    stop("Type of analysis not specified correctly. Must be 'amount' or 'delay'.")
  }
  
  # Calculate participant means
  all_means <- clean_data %>%
    filter(type %in% c("itc", "amt_binary", "delay_binary", "numeracy", "numberline")) %>% # filter quantitative data
    mutate(response = parse_integer(response)) %>% # convert responses to integers
    group_by(date, condition, participant, type) %>% # for each condition, participant, and type
    summarize(mean = mean(response),
              sum = sum(response),
              n = length(response), .groups = "drop") %>% # calculate mean response
    pivot_wider(names_from = type, values_from = c(mean, sum, n)) %>% # create separate columns for quantitative variables
    # filter(amt_binary <= 0.95 & amt_binary >= 0.05 & delay_binary <= 0.95 & delay_binary >= 0.05) %>%   # remove participants with extreme judgments
    mutate(across(any_of(c("mean_amt_binary", "mean_delay_binary", "mean_itc")), ~ .x * 100)) %>%
    arrange(date) # sort rows by date
  
  # Remove participants with extreme judgments
  good_participants <- (all_means$participant) # extract participants without extreme judgments
  good_participant_data <- filter(clean_data, participant %in% good_participants) # keep only participants without extreme judgments
  
  # Trim to equal numbers of participants for means
  min_participants <- min(table(all_means$condition))
  if (type == "delay_mturk") {
    all_means_trimmed <- all_means %>%
      arrange(date) %>%
      group_by(condition) %>%
      slice_head(n = min_participants) %>% 
      dplyr::select(-mean_numeracy) %>% 
      dplyr::select(date:sum_itc, numeracy = sum_numeracy, n_delay_binary, n_itc) %>% 
      rename_with(~gsub("mean_", "", .x))
  } else {
    all_means_trimmed <- all_means %>%
      arrange(date) %>%
      group_by(condition) %>%
      slice_head(n = min_participants) %>% 
      dplyr::select(-mean_numeracy) %>% 
      dplyr::select(date:sum_itc, numeracy = sum_numeracy, n_amt_binary, n_delay_binary, n_itc) %>% 
      rename_with(~gsub("mean_", "", .x))
  }
  
  # Trim participants for trial data
  included_participants <- all_means_trimmed$participant
  trial_data <- filter(clean_data, participant %in% included_participants) # %>%   # keep only participants included
  
  # Convert participant means to wide format
  if (type %in% c("amount_pilot", "amount")) {
    cents_means <- all_means_trimmed %>%
      filter(condition == "cents") %>% # filter cents condition
      ungroup() %>%
      dplyr::select(amt_cents = amt_binary, delay_cents = delay_binary, itc_cents = itc)
    dollars_means <- all_means_trimmed %>%
      filter(condition == "dollars") %>% # filter dollars condition
      ungroup() %>%
      dplyr::select(amt_dollars = amt_binary, delay_dollars = delay_binary, itc_dollars = itc)
    all_means_wide <- bind_cols(cents_means, dollars_means)
  } else if (type == "delay") {
    days_means <- all_means_trimmed %>%
      filter(condition == "days") %>% # filter days condition
      ungroup() %>%
      dplyr::select(amt_days = amt_binary, delay_days = delay_binary, itc_days = itc)
    weeks_means <- all_means_trimmed %>%
      filter(condition == "weeks") %>% # filter weeks condition
      ungroup() %>%
      dplyr::select(amt_weeks = amt_binary, delay_weeks = delay_binary, itc_weeks = itc)
    all_means_wide <- bind_cols(days_means, weeks_means)
  } else if (type == "delay_mturk") {
    days_means <- all_means_trimmed %>%
      filter(condition == "days") %>% # filter days condition
      ungroup() %>%
      dplyr::select(delay_days = delay_binary, itc_days = itc)
    weeks_means <- all_means_trimmed %>%
      filter(condition == "weeks") %>% # filter weeks condition
      ungroup() %>%
      dplyr::select(delay_weeks = delay_binary, itc_weeks = itc)
    all_means_wide <- bind_cols(days_means, weeks_means)
  } else {
    stop("Type of analysis not specified correctly. Must be amount_pilot, amount, delay, or delay_mturk.")
  }
  
  # Convert trial data to wide format
  clean_data_wide <- trial_data %>%
    unite("amt_pair", smallamount:largeamount, remove = FALSE) %>% # create column for amount pair
    unite("delay_pair", shortdelay:longdelay, remove = FALSE) %>% # create column for delay pair
    unite("question", amt_pair, delay_pair, remove = FALSE, sep = "-") # create column for question
  questions <- clean_data_wide %>%
    filter(type == "itc") %>%
    group_by(question) %>%
    slice(n = 1) %>% # extract single instance of each question
    pull(question) # create vector of questions
  
  itc_wide <- clean_data_wide %>%
    filter(type == "itc")
  amount_wide <- clean_data_wide %>%
    filter(type == "amt_binary") # %>%
  delay_wide <- clean_data_wide %>%
    filter(type == "delay_binary") # %>%
  trial_wide <- itc_wide %>%
    left_join(amount_wide, by = c("experiment", "date", "participant", "condition", "smallamount", "largeamount", "amt_pair")) %>%
    dplyr::select(experiment:condition, question = question.x, amt_pair:largeamount, shortdelay = shortdelay.x, longdelay = longdelay.x, delay_pair = delay_pair.x, itc = response.x, amt_binary = response.y) %>%
    left_join(delay_wide, by = c("experiment", "date", "participant", "condition", "shortdelay", "longdelay", "delay_pair")) %>% # join delay to itc and amt
    dplyr::select(experiment:condition, question = question.x, amt_pair = amt_pair.x, delay_pair, smallamount = smallamount.x, largeamount = largeamount.x, shortdelay:longdelay, itc, amt_binary, delay_binary = response) %>%
    mutate(
      itc = as.numeric(itc),
      delay_binary = as.character(delay_binary)
    ) # convert list to character vector
  
  # Create list of objects to return
  return_list <- list("trial_data" = trial_data, "trial_wide_data" = trial_wide, "participant_means" = all_means_trimmed, "participant_means_wide" = all_means_wide)
}


## Create themes for plots
# Theme for figures without legends
theme_plots <- function() {
  theme_bw(base_size = 30, base_family = "Arial") %+replace%
    theme(
      panel.grid = element_blank(),
      legend.position = "none"
    )
}

# Theme for figures with legend with customized location
theme_legend <- function() {
  theme_bw(base_size = 30, base_family = "Arial") %+replace%
    theme(
      panel.grid = element_blank(),
      legend.title = element_blank(),
      # legend.text = element_text(size = 15),
      legend.key = element_rect(fill = "transparent", color = NA), # make legend background transparent
      legend.background = element_rect(fill = "transparent", color = NA), # make legend background transparent
      # plot.margin = margin(t = 0, r = 10, b = 0, l = 2, "mm")
    )
}

# Theme for figures with legend on side
theme_sidelegend <- function() {
  theme_bw(base_size = 25, base_family = "Arial") %+replace%
    theme(
      panel.grid = element_blank(), # remove grid lines
      legend.title = element_blank(), # remove legend title
      legend.key.width = unit(3, "line"), # increase width of legend lines
      legend.key = element_rect(fill = "transparent", color = NA), # make legend background transparent
      legend.background = element_rect(fill = "transparent", color = NA) # make legend background transparent
    )
}


# General data processing ------------------------------------------------------

# Define color-blind safe colors
cb_palette_black <- c("#E69F00", "#56B4E9", "#009E73", "#F0E442", "#0072B2", "#D55E00", "#CC79A7", "#000000")

# Import data
all_data <- read_csv("stevens_etal_2021_data.csv")


# ITC Amount UNL 1 (In-person) ------------------------------------------------

## Data cleaning ---------------------------------------------------------

## Import and clean data
all_data_amt_unl1 <- filter(all_data, experiment == "amount_unl1") %>% # read data
  mutate(issues = NA)

cleaned_data_amt_unl1 <- process_itc_data(all_data_amt_unl1, "amount") # clean data
trial_data_amt_unl1_all <- cleaned_data_amt_unl1$trial_data # assign clean data
trial_data_wide_amt_unl1_all <- cleaned_data_amt_unl1$trial_wide_data # assign clean data
all_means_amt_unl1_all <- cleaned_data_amt_unl1$participant_means # assign participant means
all_means_wide_amt_unl1_all <- cleaned_data_amt_unl1$participant_means_wide # assign wide data

## Select first 56 participants (based on sequential sampling of BFs)
all_means_amt_unl1 <- all_means_amt_unl1_all %>%
  group_by(condition) %>%
  slice_head(n = 56)
first_participants_amt_unl1 <- pull(all_means_amt_unl1, participant)
all_means_wide_amt_unl1 <- all_means_wide_amt_unl1_all %>%
  slice_head(n = 56)
trial_data_amt_unl1 <- trial_data_amt_unl1_all %>%
  filter(participant %in% first_participants_amt_unl1)
trial_data_wide_amt_unl1 <- trial_data_wide_amt_unl1_all %>%
  filter(participant %in% first_participants_amt_unl1)


## Condition effects on similarity and ITC ---------------------------------------------------------

# Intertemporal choice
## Conduct t-tests
itc_cents_dollars_ttest_amt_unl1 <- t.test(all_means_wide_amt_unl1$itc_cents, all_means_wide_amt_unl1$itc_dollars) # conduct frequentist t-test
itc_cents_dollars_ttest_bf_amt_unl1 <- ttestBF(all_means_wide_amt_unl1$itc_cents, all_means_wide_amt_unl1$itc_dollars) # conduct Bayesian t-test

## Plot raincloud plot
ggplot(data = all_means_amt_unl1, aes(y = itc, x = condition)) +
  stat_dots(aes(fill = condition, color = condition), side = "left", binwidth = 1.5, justification = 1.05) +
  stat_slab(aes(fill = condition, size = 1), alpha = 0.5, scale = 0.7) +
  stat_summary(fun.data = mean_cl_normal, position = position_nudge(x = 0.2), size = 1) +
  geom_boxplot(position = position_nudge(x = 0.05, y = 0), width = 0.1, outlier.shape = NA, alpha = 0.5, color = "black") +
  scale_color_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  scale_fill_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  labs(x = "Condition", y = "Percent larger, later") + # label axes
  theme_plots() # use plot theme
ggsave("figures/itc_amt_unl1.png", width = 6, height = 5) # save plot

# Amount similarity judgments
## Conduct t-tests
amt_cents_dollars_ttest_amt_unl1 <- t.test(all_means_wide_amt_unl1$amt_dollars, all_means_wide_amt_unl1$amt_cents) # conduct frequentist t-test
amt_cents_dollars_ttest_bf_amt_unl1 <- ttestBF(all_means_wide_amt_unl1$amt_dollars, all_means_wide_amt_unl1$amt_cents) # conduct Bayesian t-test

## Plot raincloud plot
ggplot(data = all_means_amt_unl1, aes(y = amt_binary, x = condition)) +
  stat_dots(aes(fill = condition, color = condition), side = "left", binwidth = 1.5, justification = 1.05) +
  stat_slab(aes(fill = condition, size = 1), alpha = 0.5, scale = 0.7) +
  stat_summary(fun.data = mean_cl_normal, position = position_nudge(x = 0.2), size = 1) +
  geom_boxplot(position = position_nudge(x = 0.05, y = 0), width = 0.1, outlier.shape = NA, alpha = 0.5, color = "black") +
  scale_color_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  scale_fill_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  labs(x = "Condition", y = "Percent similar") + # label axes
  theme_plots() # use plot theme
ggsave("figures/amount_amt_unl1.png", width = 6, height = 5) # save plot


## Similarity effects on intertemporal choice ---------------------------------------------------------

all_amt_subject_means_unl1 <- trial_data_wide_amt_unl1 %>%
  group_by(participant, amt_binary, condition) %>%
  summarise(itc = mean(itc), .groups = "drop") %>%
  mutate(amt_binary_fac = fct_recode(amt_binary, "Similar" = "1", "Dissimilar" = "0"),
         amt_binary = as.numeric(amt_binary))

ggplot(all_amt_subject_means_unl1, aes(x = condition, y = itc * 100, group = amt_binary_fac, color = amt_binary_fac, shape = amt_binary_fac)) +
  stat_summary(fun.data = "mean_cl_boot", position = position_dodge(width = 0.1), size = 1) +
  stat_summary(fun = "mean", geom = "line", position = position_dodge(width = 0.1), size = 1) +
  scale_color_manual(values = cb_palette_black[c(2, 6)]) + # set group color
  labs(x = "Amount judgment", y = "Percent larger, later") +
  guides(color = guide_legend(override.aes = list(linetype = 0))) +
  theme_legend() +
  theme(legend.position = c(0.75, 0.9)) # adjust font size
ggsave("figures/amt_itc_amt_unl1.png", width = 6, height = 5)

# Mediation analysis
trial_data_wide_amt_unl1$amt_binary <- as.numeric(trial_data_wide_amt_unl1$amt_binary)
mediator_model_amt_unl1 <- glmer(amt_binary ~ condition + (1 | participant), data = trial_data_wide_amt_unl1, family = binomial())
full_model_amt_unl1 <- glmer(itc ~ condition + amt_binary + (1 | participant), data = trial_data_wide_amt_unl1, family = binomial())
mediate_amt_unl1 <- mediate(mediator_model_amt_unl1, full_model_amt_unl1, treat = "condition", mediator = "amt_binary")
mediate_summary_amt_unl1 <- summary(mediate_amt_unl1)


## Numeracy and numberline ---------------------------------------------------------

# Numeracy and numberline
trimmed_means_amt_unl1 <- all_means_amt_unl1 %>% 
  filter(numberline < 100) %>% 
  mutate(log_number = log(numberline))
numerical_corr_amt_unl1 <- cor.test(trimmed_means_amt_unl1$numberline, trimmed_means_amt_unl1$numeracy)
numerical_corr_bf_amt_unl1 <- correlationBF(trimmed_means_amt_unl1$numberline, trimmed_means_amt_unl1$numeracy)

# Numeracy
## Similarity judgments
amt_intercept_amt_unl1 <- lm(amt_binary ~ 1, data = trimmed_means_amt_unl1)
amt_numeracy_amt_unl1 <- lm(amt_binary ~ numeracy, data = trimmed_means_amt_unl1)
amt_condition_amt_unl1 <- lm(amt_binary ~ condition, data = trimmed_means_amt_unl1)
amt_numeracy_cond_amt_unl1 <- lm(amt_binary ~ condition + numeracy, data = trimmed_means_amt_unl1)
amt_numeracy_full_amt_unl1 <- lm(amt_binary ~ condition * numeracy, data = trimmed_means_amt_unl1)

(comparison_amt_numeracy_amt_unl1 <- bf_models(amt_intercept_amt_unl1, amt_numeracy_amt_unl1, amt_condition_amt_unl1, amt_numeracy_cond_amt_unl1, amt_numeracy_full_amt_unl1))

## Intertemporal choice
itc_intercept_amt_unl1 <- lm(itc ~ 1, data = trimmed_means_amt_unl1)
itc_numeracy_amt_unl1 <- lm(itc ~ numeracy, data = trimmed_means_amt_unl1)
itc_condition_amt_unl1 <- lm(itc ~ condition, data = trimmed_means_amt_unl1)
itc_numeracy_cond_amt_unl1 <- lm(itc ~ condition + numeracy, data = trimmed_means_amt_unl1)
itc_numeracy_full_amt_unl1 <- lm(itc ~ condition * numeracy, data = trimmed_means_amt_unl1)

(comparison_itc_numeracy_amt_unl1 <- bf_models(itc_intercept_amt_unl1, itc_numeracy_amt_unl1, itc_condition_amt_unl1, itc_numeracy_cond_amt_unl1, itc_numeracy_full_amt_unl1))

# Numberline
## Similarity judgments
amt_number_amt_unl1 <- lm(amt_binary ~ log_number, data = trimmed_means_amt_unl1)
amt_number_cond_amt_unl1 <- lm(amt_binary ~ condition + log_number, data = trimmed_means_amt_unl1)
amt_number_full_amt_unl1 <- lm(amt_binary ~ condition * log_number, data = trimmed_means_amt_unl1)

(comparison_amt_number_amt_unl1 <- bf_models(amt_intercept_amt_unl1, amt_number_amt_unl1, amt_condition_amt_unl1, amt_number_cond_amt_unl1, amt_number_full_amt_unl1))

## Intertemporal choice
itc_number_amt_unl1 <- lm(itc ~ log_number, data = trimmed_means_amt_unl1)
itc_number_cond_amt_unl1 <- lm(itc ~ condition + log_number, data = trimmed_means_amt_unl1)
itc_number_full_amt_unl1 <- lm(itc ~ condition * log_number, data = trimmed_means_amt_unl1)
(comparison_itc_number_amt_unl1 <- bf_models(itc_intercept_amt_unl1, itc_number_amt_unl1, itc_condition_amt_unl1, itc_number_cond_amt_unl1, itc_number_full_amt_unl1))


# ITC Amount UNL 2 (Online) ------------------------------------------------

## Data cleaning ---------------------------------------------------------

## Import and clean data
all_data_amt_unl2 <- filter(all_data, experiment == "amount_unl2") %>% # read data
  mutate(issues = NA)

cleaned_data_amt_unl2 <- process_itc_data(all_data_amt_unl2, "amount") # clean data
trial_data_amt_unl2_all <- cleaned_data_amt_unl2$trial_data # assign clean data
trial_data_wide_amt_unl2_all <- cleaned_data_amt_unl2$trial_wide_data # assign clean data
all_means_amt_unl2_all <- cleaned_data_amt_unl2$participant_means # assign participant means
all_means_wide_amt_unl2_all <- cleaned_data_amt_unl2$participant_means_wide # assign wide data

## Select first 49 participants (based on sequential sampling of BFs)
all_means_amt_unl2 <- all_means_amt_unl2_all %>%
  group_by(condition) %>%
  slice_head(n = 49)
first_participants_amt_unl2 <- pull(all_means_amt_unl2, participant)
all_means_wide_amt_unl2 <- all_means_wide_amt_unl2_all %>%
  slice_head(n = 49)
trial_data_amt_unl2 <- trial_data_amt_unl2_all %>%
  filter(participant %in% first_participants_amt_unl2)
trial_data_wide_amt_unl2 <- trial_data_wide_amt_unl2_all %>%
  filter(participant %in% first_participants_amt_unl2)


## Condition effects on similarity and ITC ---------------------------------------------------------

# Intertemporal choice
## Conduct t-tests
itc_cents_dollars_ttest_amt_unl2 <- t.test(all_means_wide_amt_unl2$itc_cents, all_means_wide_amt_unl2$itc_dollars) # conduct frequentist t-test
itc_cents_dollars_ttest_bf_amt_unl2 <- ttestBF(all_means_wide_amt_unl2$itc_cents, all_means_wide_amt_unl2$itc_dollars) # conduct Bayesian t-test

## Plot raincloud plot
ggplot(data = all_means_amt_unl2, aes(y = itc, x = condition)) +
  stat_dots(aes(fill = condition, color = condition), side = "left", binwidth = 1.5, justification = 1.05) +
  stat_slab(aes(fill = condition, size = 1), alpha = 0.5, scale = 0.7) +
  stat_summary(fun.data = mean_cl_normal, position = position_nudge(x = 0.2), size = 1) +
  geom_boxplot(position = position_nudge(x = 0.05, y = 0), width = 0.1, outlier.shape = NA, alpha = 0.5, color = "black") +
  scale_color_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  scale_fill_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  labs(x = "Condition", y = "Percent larger, later") + # label axes
  theme_plots() # use plot theme
ggsave("figures/itc_amt_unl2.png", width = 6, height = 5) # save plot

# Amount similarity judgments
## Conduct t-tests
amt_cents_dollars_ttest_amt_unl2 <- t.test(all_means_wide_amt_unl2$amt_dollars, all_means_wide_amt_unl2$amt_cents) # conduct frequentist t-test
amt_cents_dollars_ttest_bf_amt_unl2 <- ttestBF(all_means_wide_amt_unl2$amt_dollars, all_means_wide_amt_unl2$amt_cents) # conduct Bayesian t-test

## Plot raincloud plot
ggplot(data = all_means_amt_unl2, aes(y = amt_binary, x = condition)) +
  stat_dots(aes(fill = condition, color = condition), side = "left", binwidth = 1.5, justification = 1.05) +
  stat_slab(aes(fill = condition, size = 1), alpha = 0.5, scale = 0.7) +
  stat_summary(fun.data = mean_cl_normal, position = position_nudge(x = 0.2), size = 1) +
  geom_boxplot(position = position_nudge(x = 0.05, y = 0), width = 0.1, outlier.shape = NA, alpha = 0.5, color = "black") +
  scale_color_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  scale_fill_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  labs(x = "Condition", y = "Percent similar") + # label axes
  theme_plots() # use plot theme
ggsave("figures/amount_amt_unl2.png", width = 6, height = 5) # save plot


## Similarity effects on intertemporal choice ---------------------------------------------------------

all_amt_subject_means_unl2 <- trial_data_wide_amt_unl2 %>%
  group_by(participant, amt_binary, condition) %>%
  summarise(itc = mean(itc), .groups = "drop") %>%
  mutate(amt_binary_fac = fct_recode(amt_binary, "Similar" = "1", "Dissimilar" = "0"),
         amt_binary = as.numeric(amt_binary))

ggplot(all_amt_subject_means_unl2, aes(x = condition, y = itc * 100, group = amt_binary_fac, color = amt_binary_fac, shape = amt_binary_fac)) +
  stat_summary(fun.data = "mean_cl_boot", position = position_dodge(width = 0.1), size = 1) +
  stat_summary(fun = "mean", geom = "line", position = position_dodge(width = 0.1), size = 1) +
  scale_color_manual(values = cb_palette_black[c(2, 6)]) + # set group color
  labs(x = "Amount judgment", y = "Percent larger, later") +
  guides(color = guide_legend(override.aes = list(linetype = 0))) +
  theme_legend() +
  theme(legend.position = c(0.75, 0.9)) # adjust font size
ggsave("figures/amt_itc_amt_unl2.png", width = 6, height = 5)

# Mediation analysis
trial_data_wide_amt_unl2$amt_binary <- as.numeric(trial_data_wide_amt_unl2$amt_binary)
mediator_model_amt_unl2 <- glmer(amt_binary ~ condition + (1 | participant), data = trial_data_wide_amt_unl2, family = binomial())
full_model_amt_unl2 <- glmer(itc ~ condition + amt_binary + (1 | participant), data = trial_data_wide_amt_unl2, family = binomial())
mediate_amt_unl2 <- mediate(mediator_model_amt_unl2, full_model_amt_unl2, treat = "condition", mediator = "amt_binary")
mediate_summary_amt_unl2 <- summary(mediate_amt_unl2)


## Numeracy and numberline ---------------------------------------------------------

# Numeracy and numberline
trimmed_means_amt_unl2 <- all_means_amt_unl2 %>% 
  filter(numberline < 100) %>% 
  mutate(log_number = log(numberline))
numerical_corr_amt_unl2 <- cor.test(trimmed_means_amt_unl2$numberline, trimmed_means_amt_unl2$numeracy)
numerical_corr_bf_amt_unl2 <- correlationBF(trimmed_means_amt_unl2$numberline, trimmed_means_amt_unl2$numeracy)

# Numeracy
## Similarity judgments
amt_intercept_amt_unl2 <- lm(amt_binary ~ 1, data = filter(trimmed_means_amt_unl2, !is.na(numeracy)))
amt_numeracy_amt_unl2 <- lm(amt_binary ~ numeracy, data = trimmed_means_amt_unl2)
amt_condition_amt_unl2 <- lm(amt_binary ~ condition, data = trimmed_means_amt_unl2)
amt_numeracy_cond_amt_unl2 <- lm(amt_binary ~ condition + numeracy, data = trimmed_means_amt_unl2)
amt_numeracy_full_amt_unl2 <- lm(amt_binary ~ condition * numeracy, data = trimmed_means_amt_unl2)

(comparison_amt_numeracy_amt_unl2 <- bf_models(amt_intercept_amt_unl2, amt_numeracy_amt_unl2, amt_condition_amt_unl2, amt_numeracy_cond_amt_unl2, amt_numeracy_full_amt_unl2))

## Intertemporal choice
itc_intercept_amt_unl2 <- lm(itc ~ 1, data = filter(trimmed_means_amt_unl2, !is.na(numeracy)))
itc_numeracy_amt_unl2 <- lm(itc ~ numeracy, data = trimmed_means_amt_unl2)
itc_condition_amt_unl2 <- lm(itc ~ condition, data = trimmed_means_amt_unl2)
itc_numeracy_cond_amt_unl2 <- lm(itc ~ condition + numeracy, data = trimmed_means_amt_unl2)
itc_numeracy_full_amt_unl2 <- lm(itc ~ condition * numeracy, data = trimmed_means_amt_unl2)

(comparison_itc_numeracy_amt_unl2 <- bf_models(itc_intercept_amt_unl2, itc_numeracy_amt_unl2, itc_condition_amt_unl2, itc_numeracy_cond_amt_unl2, itc_numeracy_full_amt_unl2))

# Numberline
## Similarity judgments
amt_intercept_amt_unl2 <- lm(amt_binary ~ 1, data = trimmed_means_amt_unl2)
amt_number_amt_unl2 <- lm(amt_binary ~ log_number, data = trimmed_means_amt_unl2)
amt_number_cond_amt_unl2 <- lm(amt_binary ~ condition + log_number, data = trimmed_means_amt_unl2)
amt_number_full_amt_unl2 <- lm(amt_binary ~ condition * log_number, data = trimmed_means_amt_unl2)

(comparison_amt_number_amt_unl2 <- bf_models(amt_intercept_amt_unl2, amt_number_amt_unl2, amt_condition_amt_unl2, amt_number_cond_amt_unl2, amt_number_full_amt_unl2))

## Intertemporal choice
itc_intercept_amt_unl2 <- lm(itc ~ 1, data = trimmed_means_amt_unl2)
itc_number_amt_unl2 <- lm(itc ~ log_number, data = trimmed_means_amt_unl2)
itc_number_cond_amt_unl2 <- lm(itc ~ condition + log_number, data = trimmed_means_amt_unl2)
itc_number_full_amt_unl2 <- lm(itc ~ condition * log_number, data = trimmed_means_amt_unl2)

(comparison_itc_number_amt_unl2 <- bf_models(itc_intercept_amt_unl2, itc_number_amt_unl2, itc_condition_amt_unl2, itc_number_cond_amt_unl2, itc_number_full_amt_unl2))


# ITC Delay UNL 1 (In-person) ------------------------------------------------

## Data cleaning ---------------------------------------------------------

## Import and clean data
all_data_delay_unl1 <- filter(all_data, experiment == "delay_unl1") %>% # read data
  mutate(issues = NA)

cleaned_data_delay_unl1 <- process_itc_data(all_data_delay_unl1, "delay") # clean data
trial_data_delay_unl1_all <- cleaned_data_delay_unl1$trial_data # assign clean data
trial_data_wide_delay_unl1_all <- cleaned_data_delay_unl1$trial_wide_data # assign clean data
all_means_delay_unl1_all <- cleaned_data_delay_unl1$participant_means # assign participant means
all_means_wide_delay_unl1_all <- cleaned_data_delay_unl1$participant_means_wide # assign wide data

## Select first 86 participants (based on sequential sampling of BFs)
all_means_delay_unl1 <- all_means_delay_unl1_all %>%
  group_by(condition) %>%
  slice_head(n = 86)
first_participants_delay_unl1 <- pull(all_means_delay_unl1, participant)
all_means_wide_delay_unl1 <- all_means_wide_delay_unl1_all %>%
  slice_head(n = 86)
trial_data_delay_unl1 <- trial_data_delay_unl1_all %>%
  filter(participant %in% first_participants_delay_unl1)
trial_data_wide_delay_unl1 <- trial_data_wide_delay_unl1_all %>%
  filter(participant %in% first_participants_delay_unl1)


## Condition effects on similarity and ITC ---------------------------------------------------------

# Intertemporal choice
## Conduct t-tests
itc_days_weeks_ttest_delay_unl1 <- t.test(all_means_wide_delay_unl1$itc_weeks, all_means_wide_delay_unl1$itc_days) # conduct frequentist t-test
itc_days_weeks_ttest_bf_delay_unl1 <- ttestBF(all_means_wide_delay_unl1$itc_weeks, all_means_wide_delay_unl1$itc_days) # conduct Bayesian t-test

## Plot raincloud plot
ggplot(data = all_means_delay_unl1, aes(y = itc, x = condition)) +
  stat_dots(aes(fill = condition, color = condition), side = "left", binwidth = 1.5, justification = 1.05) +
  stat_slab(aes(fill = condition, size = 1), alpha = 0.5, scale = 0.7) +
  stat_summary(fun.data = mean_cl_normal, position = position_nudge(x = 0.2), size = 1) +
  geom_boxplot(position = position_nudge(x = 0.05, y = 0), width = 0.1, outlier.shape = NA, alpha = 0.5, color = "black") +
  scale_color_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  scale_fill_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  labs(x = "Condition", y = "Percent larger, later") + # label axes
  theme_plots() # use plot theme
ggsave("figures/itc_delay_unl1.png", width = 6, height = 5) # save plot

# Delay similarity judgments
## Conduct t-tests
delay_days_weeks_ttest_delay_unl1 <- t.test(all_means_wide_delay_unl1$delay_weeks, all_means_wide_delay_unl1$delay_days) # conduct frequentist t-test
delay_days_weeks_ttest_bf_delay_unl1 <- ttestBF(all_means_wide_delay_unl1$delay_weeks, all_means_wide_delay_unl1$delay_days) # conduct Bayesian t-test

## Plot raincloud plot
ggplot(data = all_means_delay_unl1, aes(y = delay_binary, x = condition)) +
  stat_dots(aes(fill = condition, color = condition), side = "left", binwidth = 1.5, justification = 1.05) +
  stat_slab(aes(fill = condition, size = 1), alpha = 0.5, scale = 0.7) +
  stat_summary(fun.data = mean_cl_normal, position = position_nudge(x = 0.2), size = 1) +
  geom_boxplot(position = position_nudge(x = 0.05, y = 0), width = 0.1, outlier.shape = NA, alpha = 0.5, color = "black") +
  scale_color_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  scale_fill_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  labs(x = "Condition", y = "Percent similar") + # label axes
  # geom_text(aes(x = 1.5, y = 0.96, label = paste("BF =", round(extractBF(delay_days_weeks_ttest_bf_delay_unl1)$bf, 2))), size = 8, color = "black") + # add Bayes factor
  theme_plots() # use plot theme
ggsave("figures/delay_delay_unl1.png", width = 6, height = 5) # save plot


## Similarity effects on intertemporal choice ---------------------------------------------------------

all_delay_subject_means_unl1 <- trial_data_wide_delay_unl1 %>%
  filter(delay_binary != "NULL") %>%
  group_by(participant, delay_binary, condition) %>%
  summarise(itc = mean(itc), .groups = "drop") %>%
  mutate(delay_binary_fac = fct_recode(delay_binary, "Similar" = "1", "Dissimilar" = "0"),
         delay_binary = as.numeric(delay_binary))

ggplot(all_delay_subject_means_unl1, aes(x = condition, y = itc * 100, group = delay_binary_fac, color = delay_binary_fac, shape = delay_binary_fac)) +
  stat_summary(fun.data = "mean_cl_boot", position = position_dodge(width = 0.1), size = 1) +
  stat_summary(fun = "mean", geom = "line", position = position_dodge(width = 0.1), size = 1) +
  scale_color_manual(values = cb_palette_black[c(2, 6)]) + # set group color
  labs(x = "Delay judgment", y = "Percent larger, later") +
  guides(color = guide_legend(override.aes = list(linetype = 0))) +
  theme_legend() +
  theme(legend.position = c(0.25, 0.9)) # adjust font size
ggsave("figures/delay_itc_delay_unl1.png", width = 6, height = 5)

# Mediation analysis
trial_data_wide_delay_unl1$delay_binary <- as.numeric(trial_data_wide_delay_unl1$delay_binary)
mediator_model_delay_unl1 <- glmer(delay_binary ~ condition + (1 | participant), data = trial_data_wide_delay_unl1, family = binomial())
full_model_delay_unl1 <- glmer(itc ~ condition + delay_binary + (1 | participant), data = trial_data_wide_delay_unl1, family = binomial())
mediate_delay_unl1 <- mediate(mediator_model_delay_unl1, full_model_delay_unl1, treat = "condition", mediator = "delay_binary")
mediate_summary_delay_unl1 <- summary(mediate_delay_unl1)


## Numeracy and numberline ---------------------------------------------------------

# Numeracy and numberline
trimmed_means_delay_unl1 <- all_means_delay_unl1 %>% 
  filter(numberline < 100) %>% 
  mutate(log_number = log(numberline))
numerical_corr_delay_unl1 <- cor.test(trimmed_means_delay_unl1$numberline, trimmed_means_delay_unl1$numeracy)
numerical_corr_bf_delay_unl1 <- correlationBF(trimmed_means_delay_unl1$numberline, trimmed_means_delay_unl1$numeracy)

# Numeracy
## Similarity judgments
delay_intercept_delay_unl1 <- lm(delay_binary ~ 1, data = trimmed_means_delay_unl1)
delay_numeracy_delay_unl1 <- lm(delay_binary ~ numeracy, data = trimmed_means_delay_unl1)
delay_condition_delay_unl1 <- lm(delay_binary ~ condition, data = trimmed_means_delay_unl1)
delay_numeracy_cond_delay_unl1 <- lm(delay_binary ~ condition + numeracy, data = trimmed_means_delay_unl1)
delay_numeracy_full_delay_unl1 <- lm(delay_binary ~ condition * numeracy, data = trimmed_means_delay_unl1)

(comparison_delay_numeracy_delay_unl1 <- bf_models(delay_intercept_delay_unl1, delay_numeracy_delay_unl1, delay_condition_delay_unl1, delay_numeracy_cond_delay_unl1, delay_numeracy_full_delay_unl1))

## Intertemporal choice
itc_intercept_delay_unl1 <- lm(itc ~ 1, data = trimmed_means_delay_unl1)
itc_numeracy_delay_unl1 <- lm(itc ~ numeracy, data = trimmed_means_delay_unl1)
itc_condition_delay_unl1 <- lm(itc ~ condition, data = trimmed_means_delay_unl1)
itc_numeracy_cond_delay_unl1 <- lm(itc ~ condition + numeracy, data = trimmed_means_delay_unl1)
itc_numeracy_full_delay_unl1 <- lm(itc ~ condition * numeracy, data = trimmed_means_delay_unl1)

(comparison_itc_numeracy_delay_unl1 <- bf_models(itc_intercept_delay_unl1, itc_numeracy_delay_unl1, itc_condition_delay_unl1, itc_numeracy_cond_delay_unl1, itc_numeracy_full_delay_unl1))

# Numberline
## Similarity judgments
delay_number_delay_unl1 <- lm(delay_binary ~ log_number, data = trimmed_means_delay_unl1)
delay_condition_delay_unl1 <- lm(delay_binary ~ condition, data = trimmed_means_delay_unl1)
delay_number_cond_delay_unl1 <- lm(delay_binary ~ condition + log_number, data = trimmed_means_delay_unl1)
delay_number_full_delay_unl1 <- lm(delay_binary ~ condition * log_number, data = trimmed_means_delay_unl1)

(comparison_delay_number_delay_unl1 <- bf_models(delay_intercept_delay_unl1, delay_number_delay_unl1, delay_condition_delay_unl1, delay_number_cond_delay_unl1, delay_number_full_delay_unl1))

## Intertemporal choice
itc_number_delay_unl1 <- lm(itc ~ log_number, data = trimmed_means_delay_unl1)
itc_number_cond_delay_unl1 <- lm(itc ~ condition + log_number, data = trimmed_means_delay_unl1)
itc_number_full_delay_unl1 <- lm(itc ~ condition * log_number, data = trimmed_means_delay_unl1)

(comparison_itc_number_delay_unl1 <- bf_models(itc_intercept_delay_unl1, itc_number_delay_unl1, itc_condition_delay_unl1, itc_number_cond_delay_unl1, itc_number_full_delay_unl1))


# ITC Delay UNL 2 (Online) ------------------------------------------------

## Data cleaning ---------------------------------------------------------

## Import and clean data
all_data_delay_unl2 <- filter(all_data, experiment == "delay_unl2") %>% # read data
  mutate(issues = NA)

cleaned_data_delay_unl2 <- process_itc_data(all_data_delay_unl2, "delay") # clean data
trial_data_delay_unl2_all <- cleaned_data_delay_unl2$trial_data # assign clean data
trial_data_wide_delay_unl2_all <- cleaned_data_delay_unl2$trial_wide_data # assign clean data
all_means_delay_unl2_all <- cleaned_data_delay_unl2$participant_means # assign participant means
all_means_wide_delay_unl2_all <- cleaned_data_delay_unl2$participant_means_wide # assign wide data

## Select first 46 participants (based on sequential sampling of BFs)
all_means_delay_unl2 <- all_means_delay_unl2_all %>%
  group_by(condition) %>%
  slice_head(n = 46)
first_participants_delay_unl2 <- pull(all_means_delay_unl2, participant)
all_means_wide_delay_unl2 <- all_means_wide_delay_unl2_all %>%
  slice_head(n = 46)
trial_data_delay_unl2 <- trial_data_delay_unl2_all %>%
  filter(participant %in% first_participants_delay_unl2)
trial_data_wide_delay_unl2 <- trial_data_wide_delay_unl2_all %>%
  filter(participant %in% first_participants_delay_unl2)


## Condition effects on similarity and ITC ---------------------------------------------------------

# Intertemporal choice
## Conduct t-tests
itc_days_weeks_ttest_delay_unl2 <- t.test(all_means_wide_delay_unl2$itc_weeks, all_means_wide_delay_unl2$itc_days) # conduct frequentist t-test
itc_days_weeks_ttest_bf_delay_unl2 <- ttestBF(all_means_wide_delay_unl2$itc_weeks, all_means_wide_delay_unl2$itc_days) # conduct Bayesian t-test

## Plot raincloud plot
ggplot(data = all_means_delay_unl2, aes(y = itc, x = condition)) +
  stat_dots(aes(fill = condition, color = condition), side = "left", binwidth = 1.5, justification = 1.05) +
  stat_slab(aes(fill = condition, size = 1), alpha = 0.5, scale = 0.7) +
  stat_summary(fun.data = mean_cl_normal, position = position_nudge(x = 0.2), size = 1) +
  geom_boxplot(position = position_nudge(x = 0.05, y = 0), width = 0.1, outlier.shape = NA, alpha = 0.5, color = "black") +
  scale_color_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  scale_fill_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  labs(x = "Condition", y = "Percent larger, later") + # label axes
  theme_plots() # use plot theme
ggsave("figures/itc_delay_unl2.png", width = 6, height = 5) # save plot

# Delay similarity judgments
## Conduct t-tests
delay_days_weeks_ttest_delay_unl2 <- t.test(all_means_wide_delay_unl2$delay_weeks, all_means_wide_delay_unl2$delay_days) # conduct frequentist t-test
delay_days_weeks_ttest_bf_delay_unl2 <- ttestBF(all_means_wide_delay_unl2$delay_weeks, all_means_wide_delay_unl2$delay_days) # conduct Bayesian t-test

## Plot raincloud plot
ggplot(data = all_means_delay_unl2, aes(y = delay_binary, x = condition)) +
  stat_dots(aes(fill = condition, color = condition), side = "left", binwidth = 1.5, justification = 1.05) +
  stat_slab(aes(fill = condition, size = 1), alpha = 0.5, scale = 0.7) +
  stat_summary(fun.data = mean_cl_normal, position = position_nudge(x = 0.2), size = 1) +
  geom_boxplot(position = position_nudge(x = 0.05, y = 0), width = 0.1, outlier.shape = NA, alpha = 0.5, color = "black") +
  scale_color_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  scale_fill_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  labs(x = "Condition", y = "Percent similar") + # label axes
  theme_plots() # use plot theme
ggsave("figures/delay_delay_unl2.png", width = 6, height = 5) # save plot


## Similarity effects on intertemporal choice ---------------------------------------------------------

all_delay_subject_means_unl2 <- trial_data_wide_delay_unl2 %>%
  filter(delay_binary != "NULL") %>%
  group_by(participant, delay_binary, condition) %>%
  summarise(itc = mean(itc), .groups = "drop") %>%
  mutate(delay_binary_fac = fct_recode(delay_binary, "Similar" = "1", "Dissimilar" = "0"),
         delay_binary = as.numeric(delay_binary))

ggplot(all_delay_subject_means_unl2, aes(x = condition, y = itc * 100, group = delay_binary_fac, color = delay_binary_fac, shape = delay_binary_fac)) +
  stat_summary(fun.data = "mean_cl_boot", position = position_dodge(width = 0.1), size = 1) +
  stat_summary(fun = "mean", geom = "line", position = position_dodge(width = 0.1), size = 1) +
  scale_color_manual(values = cb_palette_black[c(2, 6)]) + # set group color
  labs(x = "Delay judgment", y = "Percent larger, later") +
  guides(color = guide_legend(override.aes = list(linetype = 0))) +
  theme_legend() +
  theme(legend.position = c(0.75, 0.15)) # adjust font size
ggsave("figures/delay_itc_delay_unl2.png", width = 6, height = 5)

# Mediation analysis
trial_data_wide_delay_unl2$delay_binary <- as.numeric(trial_data_wide_delay_unl2$delay_binary)
mediator_model_delay_unl2 <- glmer(delay_binary ~ condition + (1 | participant), data = trial_data_wide_delay_unl2, family = binomial())
full_model_delay_unl2 <- glmer(itc ~ condition + delay_binary + (1 | participant), data = trial_data_wide_delay_unl2, family = binomial())
mediate_delay_unl2 <- mediate(mediator_model_delay_unl2, full_model_delay_unl2, treat = "condition", mediator = "delay_binary")
mediate_summary_delay_unl2 <- summary(mediate_delay_unl2)


## Numeracy and numberline ---------------------------------------------------------

# Numeracy and numberline
trimmed_means_delay_unl2 <- all_means_delay_unl2 %>% 
  filter(numberline < 100) %>% 
  mutate(log_number = log(numberline))
numerical_corr_delay_unl2 <- cor.test(trimmed_means_delay_unl2$numberline, trimmed_means_delay_unl2$numeracy)
numerical_corr_bf_delay_unl2 <- correlationBF(trimmed_means_delay_unl2$numberline, trimmed_means_delay_unl2$numeracy)

# Numeracy
## Similarity judgments
delay_intercept_delay_unl2 <- lm(delay_binary ~ 1, data = trimmed_means_delay_unl2)
delay_numeracy_delay_unl2 <- lm(delay_binary ~ numeracy, data = trimmed_means_delay_unl2)
delay_condition_delay_unl2 <- lm(delay_binary ~ condition, data = trimmed_means_delay_unl2)
delay_numeracy_cond_delay_unl2 <- lm(delay_binary ~ condition + numeracy, data = trimmed_means_delay_unl2)
delay_numeracy_full_delay_unl2 <- lm(delay_binary ~ condition * numeracy, data = trimmed_means_delay_unl2)

(comparison_delay_numeracy_delay_unl2 <- bf_models(delay_intercept_delay_unl2, delay_numeracy_delay_unl2, delay_condition_delay_unl2, delay_numeracy_cond_delay_unl2, delay_numeracy_full_delay_unl2))

## Intertemporal choice
itc_intercept_delay_unl2 <- lm(itc ~ 1, data = trimmed_means_delay_unl2)
itc_numeracy_delay_unl2 <- lm(itc ~ numeracy, data = trimmed_means_delay_unl2)
itc_condition_delay_unl2 <- lm(itc ~ condition, data = trimmed_means_delay_unl2)
itc_numeracy_cond_delay_unl2 <- lm(itc ~ condition + numeracy, data = trimmed_means_delay_unl2)
itc_numeracy_full_delay_unl2 <- lm(itc ~ condition * numeracy, data = trimmed_means_delay_unl2)

(comparison_itc_numeracy_delay_unl2 <- bf_models(itc_intercept_delay_unl2, itc_numeracy_delay_unl2, itc_condition_delay_unl2, itc_numeracy_cond_delay_unl2, itc_numeracy_full_delay_unl2))

# Numberline
## Similarity judgments
delay_number_delay_unl2 <- lm(delay_binary ~ log_number, data = trimmed_means_delay_unl2)
delay_condition_delay_unl2 <- lm(delay_binary ~ condition, data = trimmed_means_delay_unl2)
delay_number_cond_delay_unl2 <- lm(delay_binary ~ condition + log_number, data = trimmed_means_delay_unl2)
delay_number_full_delay_unl2 <- lm(delay_binary ~ condition * log_number, data = trimmed_means_delay_unl2)

(comparison_delay_number_delay_unl2 <- bf_models(delay_intercept_delay_unl2, delay_number_delay_unl2, delay_condition_delay_unl2, delay_number_cond_delay_unl2, delay_number_full_delay_unl2))

## Intertemporal choice
itc_number_delay_unl2 <- lm(itc ~ log_number, data = trimmed_means_delay_unl2)
itc_condition_delay_unl2 <- lm(itc ~ condition, data = trimmed_means_delay_unl2)
itc_number_cond_delay_unl2 <- lm(itc ~ condition + log_number, data = trimmed_means_delay_unl2)
itc_number_full_delay_unl2 <- lm(itc ~ condition * log_number, data = trimmed_means_delay_unl2)

(comparison_itc_number_delay_unl2 <- bf_models(itc_intercept_delay_unl2, itc_number_delay_unl2, itc_condition_delay_unl2, itc_number_cond_delay_unl2, itc_number_full_delay_unl2))


# ITC Delay MTurk (Online) ------------------------------------------------

## Data cleaning ---------------------------------------------------------

## Import and clean data
all_data_delay_mturk <- filter(all_data, experiment == "delay_mturk") #%>%  # read data

cleaned_data_delay_mturk <- process_itc_data(all_data_delay_mturk, "delay_mturk") # clean data
trial_data_delay_mturk_all <- cleaned_data_delay_mturk$trial_data # assign clean data
trial_data_wide_delay_mturk_all <- cleaned_data_delay_mturk$trial_wide_data # assign clean data
all_means_delay_mturk_all <- cleaned_data_delay_mturk$participant_means # assign participant means
all_means_wide_delay_mturk_all <- cleaned_data_delay_mturk$participant_means_wide # assign wide data

## Select first 46 participants (based on sequential sampling of BFs)
all_means_delay_mturk <- all_means_delay_mturk_all %>%
  group_by(condition) %>%
  slice_head(n = 64)
first_participants_delay_mturk <- pull(all_means_delay_mturk, participant)
all_means_wide_delay_mturk <- all_means_wide_delay_mturk_all %>%
  slice_head(n = 64)
trial_data_delay_mturk <- trial_data_delay_mturk_all %>%
  filter(participant %in% first_participants_delay_mturk)
trial_data_wide_delay_mturk <- trial_data_wide_delay_mturk_all %>%
  filter(participant %in% first_participants_delay_mturk)


## Condition effects on similarity and ITC ---------------------------------------------------------

# Intertemporal choice
## Conduct t-tests
itc_days_weeks_ttest_delay_mturk <- t.test(all_means_wide_delay_mturk$itc_weeks, all_means_wide_delay_mturk$itc_days) # conduct frequentist t-test
itc_days_weeks_ttest_bf_delay_mturk <- ttestBF(all_means_wide_delay_mturk$itc_weeks, all_means_wide_delay_mturk$itc_days) # conduct Bayesian t-test

## Plot raincloud plot
ggplot(data = all_means_delay_mturk, aes(y = itc, x = condition)) +
  stat_dots(aes(fill = condition, color = condition), side = "left", binwidth = 1.5, justification = 1.05) +
  stat_slab(aes(fill = condition, size = 1), alpha = 0.5, scale = 0.7) +
  stat_summary(fun.data = mean_cl_normal, position = position_nudge(x = 0.2), size = 1) +
  geom_boxplot(position = position_nudge(x = 0.05, y = 0), width = 0.1, outlier.shape = NA, alpha = 0.5, color = "black") +
  scale_color_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  scale_fill_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  labs(x = "Condition", y = "Percent larger, later") + # label axes
  theme_plots() # use plot theme
ggsave("figures/itc_delay_mturk.png", width = 6, height = 5) # save plot

# Delay similarity judgments
## Conduct t-tests
delay_days_weeks_ttest_delay_mturk <- t.test(all_means_wide_delay_mturk$delay_weeks, all_means_wide_delay_mturk$delay_days) # conduct frequentist t-test
delay_days_weeks_ttest_bf_delay_mturk <- ttestBF(all_means_wide_delay_mturk$delay_weeks, all_means_wide_delay_mturk$delay_days) # conduct Bayesian t-test

## Plot raincloud plot
ggplot(data = all_means_delay_mturk, aes(y = delay_binary, x = condition)) +
  stat_dots(aes(fill = condition, color = condition), side = "left", binwidth = 1.5, justification = 1.05) +
  stat_slab(aes(fill = condition, size = 1), alpha = 0.5, scale = 0.7) +
  stat_summary(fun.data = mean_cl_normal, position = position_nudge(x = 0.2), size = 1) +
  geom_boxplot(position = position_nudge(x = 0.05, y = 0), width = 0.1, outlier.shape = NA, alpha = 0.5, color = "black") +
  scale_color_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  scale_fill_manual(values = cb_palette_black[c(5, 3)]) + # set group color
  labs(x = "Condition", y = "Percent similar") + # label axes
  theme_plots() # use plot theme
ggsave("figures/delay_delay_mturk.png", width = 6, height = 5) # save plot


## Similarity effects on intertemporal choice ---------------------------------------------------------

all_delay_subject_means_mturk <- trial_data_wide_delay_mturk %>%
  filter(delay_binary != "NULL") %>%
  group_by(participant, delay_binary, condition) %>%
  summarise(itc = mean(itc), .groups = "drop") %>%
  mutate(delay_binary_fac = fct_recode(delay_binary, "Similar" = "1", "Dissimilar" = "0"),
         delay_binary = as.numeric(delay_binary))

ggplot(all_delay_subject_means_mturk, aes(x = condition, y = itc * 100, group = delay_binary_fac, color = delay_binary_fac, shape = delay_binary_fac)) +
  stat_summary(fun.data = "mean_cl_boot", position = position_dodge(width = 0.1), size = 1) +
  stat_summary(fun = "mean", geom = "line", position = position_dodge(width = 0.1), size = 1) +
  scale_color_manual(values = cb_palette_black[c(2, 6)]) + # set group color
  labs(x = "Delay judgment", y = "Percent larger, later") +
  guides(color = guide_legend(override.aes = list(linetype = 0))) +
  theme_legend() +
  theme(legend.position = c(0.25, 0.9)) # adjust font size
ggsave("figures/delay_itc_delay_mturk.png", width = 6, height = 5)

## Mediation analysis
trial_data_wide_delay_mturk$delay_binary <- as.numeric(trial_data_wide_delay_mturk$delay_binary)
mediator_model_delay_mturk <- glmer(delay_binary ~ condition + (1 | participant), data = trial_data_wide_delay_mturk, family = binomial())
full_model_delay_mturk <- glmer(itc ~ condition + delay_binary + (1 | participant), data = trial_data_wide_delay_mturk, family = binomial())
mediate_delay_mturk <- mediate(mediator_model_delay_mturk, full_model_delay_mturk, treat = "condition", mediator = "delay_binary")
mediate_summary_delay_mturk <- summary(mediate_delay_mturk)


## Numeracy  ---------------------------------------------------------

## Similarity judgments
delay_intercept_delay_mturk <- lm(delay_binary ~ 1, data = all_means_delay_mturk)
delay_numeracy_delay_mturk <- lm(delay_binary ~ numeracy, data = all_means_delay_mturk)
delay_condition_delay_mturk <- lm(delay_binary ~ condition, data = all_means_delay_mturk)
delay_numeracy_cond_delay_mturk <- lm(delay_binary ~ condition + numeracy, data = all_means_delay_mturk)
delay_numeracy_full_delay_mturk <- lm(delay_binary ~ condition * numeracy, data = all_means_delay_mturk)

(comparison_delay_numeracy_delay_mturk <- bf_models(delay_intercept_delay_mturk, delay_numeracy_delay_mturk, delay_condition_delay_mturk, delay_numeracy_cond_delay_mturk, delay_numeracy_full_delay_mturk))

## Intertemporal choice
itc_intercept_delay_mturk <- lm(itc ~ 1, data = all_means_delay_mturk)
itc_numeracy_delay_mturk <- lm(itc ~ numeracy, data = all_means_delay_mturk)
itc_condition_delay_mturk <- lm(itc ~ condition, data = all_means_delay_mturk)
itc_numeracy_cond_delay_mturk <- lm(itc ~ condition + numeracy, data = all_means_delay_mturk)
itc_numeracy_full_delay_mturk <- lm(itc ~ condition * numeracy, data = all_means_delay_mturk)

(comparison_itc_numeracy_delay_mturk <- bf_models(itc_intercept_delay_mturk, itc_numeracy_delay_mturk, itc_condition_delay_mturk, itc_numeracy_cond_delay_mturk, itc_numeracy_full_delay_mturk))


# Supplementary materials -------------------------------------------------

# Extract demographic information from all data sets
demo_amt_unl1_all <- extract_demographics(all_data_amt_unl1)
demo_amt_unl1_clean <- extract_demographics(trial_data_amt_unl1)
demo_amt_unl2_all <- extract_demographics(all_data_amt_unl2)
demo_amt_unl2_clean <- extract_demographics(trial_data_amt_unl2)
demo_delay_unl1_all <- extract_demographics(all_data_delay_unl1)
demo_delay_unl1_clean <- extract_demographics(trial_data_delay_unl1)
demo_delay_unl2_all <- extract_demographics(all_data_delay_unl2)
demo_delay_unl2_clean <- extract_demographics(trial_data_delay_unl2)
demo_delay_mturk_all <- extract_demographics(all_data_delay_mturk, "mturk")
demo_delay_mturk_clean <- extract_demographics(trial_data_delay_mturk, "mturk")


## Amount UNL 1 (In-person) -----------------------------------------------

## Create demographics table for amount UNL 1
table_amt_unl_labels <- c("", "Conditions", "  Cents", "  Dollars", "Age", "Gender", "  Female", "  Male", "  Other", "Race", "  Asian", "  Black", "  Native American", "  White", "  Other", "Income", "  <$25,000", "  $25,001-$50,000", "  $50,001-$75,000", "  $75,001-$100,000", "  >$100,000", "  Prefer not to answer")
table_amt_unl1_clean <- c(
  paste("Study 1 (N=", demo_amt_unl1_clean$sample_size, ")", sep = ""), "",
  paste(unname(demo_amt_unl1_clean$condition["cents"]), " (", unname(demo_amt_unl1_clean$condition_percent["cents"]), ")", sep = ""),
  paste(unname(demo_amt_unl1_clean$condition["dollars"]), " (", unname(demo_amt_unl1_clean$condition_percent["dollars"]), ")", sep = ""),
  paste(round(demo_amt_unl1_clean$age_mean, 2), " (", demo_amt_unl1_clean$age_min, "-", demo_amt_unl1_clean$age_max, ")", sep = ""), "",
  paste(unname(demo_amt_unl1_clean$gender["Female"]), " (", unname(demo_amt_unl1_clean$gender_percent["Female"]), ")", sep = ""),
  paste(unname(demo_amt_unl1_clean$gender["Male"]), " (", unname(demo_amt_unl1_clean$gender_percent["Male"]), ")", sep = ""),
  paste(unname(demo_amt_unl1_clean$gender["Other"]), " (", unname(demo_amt_unl1_clean$gender_percent["Other"]), ")", sep = ""), "",
  paste(unname(demo_amt_unl1_clean$race["Asian"]), " (", unname(demo_amt_unl1_clean$race_percent["Asian"]), ")", sep = ""),
  paste(unname(demo_amt_unl1_clean$race["Black"]), " (", unname(demo_amt_unl1_clean$race_percent["Black"]), ")", sep = ""),
  paste(unname(demo_amt_unl1_clean$race["NativeAmerican"]), " (", unname(demo_amt_unl1_clean$race_percent["NativeAmerican"]), ")", sep = ""),
  paste(unname(demo_amt_unl1_clean$race["White"]), " (", unname(demo_amt_unl1_clean$race_percent["White"]), ")", sep = ""),
  paste(unname(demo_amt_unl1_clean$race["Other"]), " (", unname(demo_amt_unl1_clean$race_percent["Other"]), ")", sep = ""), "",
  paste(unname(demo_amt_unl1_clean$income["<$25000"]), " (", unname(demo_amt_unl1_clean$income_percent["<$25000"]), ")", sep = ""),
  paste(unname(demo_amt_unl1_clean$income["$25000"]), " (", unname(demo_amt_unl1_clean$income_percent["$25000"]), ")", sep = ""),
  paste(unname(demo_amt_unl1_clean$income["$50000"]), " (", unname(demo_amt_unl1_clean$income_percent["$50000"]), ")", sep = ""),
  paste(unname(demo_amt_unl1_clean$income["$75000"]), " (", unname(demo_amt_unl1_clean$income_percent["$75000"]), ")", sep = ""),
  paste(unname(demo_amt_unl1_clean$income[">$100000"]), " (", unname(demo_amt_unl1_clean$income_percent[">$100000"]), ")", sep = ""),
  paste(unname(demo_amt_unl1_clean$income["Noanswer"]), " (", unname(demo_amt_unl1_clean$income_percent["Noanswer"]), ")", sep = "")
)


## Amount UNL 2 (Online) -----------------------------------------------

## Create demographics table for amount UNL 2
table_amt_unl2_clean <- c(
  paste("Study 2 (N=", demo_amt_unl2_clean$sample_size, ")", sep = ""), "",
  paste(unname(demo_amt_unl2_clean$condition["cents"]), " (", unname(demo_amt_unl2_clean$condition_percent["cents"]), ")", sep = ""),
  paste(unname(demo_amt_unl2_clean$condition["dollars"]), " (", unname(demo_amt_unl2_clean$condition_percent["dollars"]), ")", sep = ""),
  paste(round(demo_amt_unl2_clean$age_mean, 2), " (", demo_amt_unl2_clean$age_min, "-", demo_amt_unl2_clean$age_max, ")", sep = ""), "",
  paste(unname(demo_amt_unl2_clean$gender["Female"]), " (", unname(demo_amt_unl2_clean$gender_percent["Female"]), ")", sep = ""),
  paste(unname(demo_amt_unl2_clean$gender["Male"]), " (", unname(demo_amt_unl2_clean$gender_percent["Male"]), ")", sep = ""),
  paste(unname(demo_amt_unl2_clean$gender["Other"]), " (", unname(demo_amt_unl2_clean$gender_percent["Other"]), ")", sep = ""), "",
  paste(unname(demo_amt_unl2_clean$race["Asian"]), " (", unname(demo_amt_unl2_clean$race_percent["Asian"]), ")", sep = ""),
  paste(unname(demo_amt_unl2_clean$race["Black"]), " (", unname(demo_amt_unl2_clean$race_percent["Black"]), ")", sep = ""),
  paste(unname(demo_amt_unl2_clean$race["NativeAmerican"]), " (", unname(demo_amt_unl2_clean$race_percent["NativeAmerican"]), ")", sep = ""),
  paste(unname(demo_amt_unl2_clean$race["White"]), " (", unname(demo_amt_unl2_clean$race_percent["White"]), ")", sep = ""),
  paste(unname(demo_amt_unl2_clean$race["Other"]), " (", unname(demo_amt_unl2_clean$race_percent["Other"]), ")", sep = ""), "",
  paste(unname(demo_amt_unl2_clean$income["<$25000"]), " (", unname(demo_amt_unl2_clean$income_percent["<$25000"]), ")", sep = ""),
  paste(unname(demo_amt_unl2_clean$income["$25000"]), " (", unname(demo_amt_unl2_clean$income_percent["$25000"]), ")", sep = ""),
  paste(unname(demo_amt_unl2_clean$income["$50000"]), " (", unname(demo_amt_unl2_clean$income_percent["$50000"]), ")", sep = ""),
  paste(unname(demo_amt_unl2_clean$income["$75000"]), " (", unname(demo_amt_unl2_clean$income_percent["$75000"]), ")", sep = ""),
  paste(unname(demo_amt_unl2_clean$income[">$100000"]), " (", unname(demo_amt_unl2_clean$income_percent[">$100000"]), ")", sep = ""),
  paste(unname(demo_amt_unl2_clean$income["Noanswer"]), " (", unname(demo_amt_unl2_clean$income_percent["Noanswer"]), ")", sep = "")
)

## Delay UNL 1 (In-person) -----------------------------------------------

## Create demographics table for delay UNL 1
table_delay_unl_labels <- c("", "Conditions", "  Days", "  Weeks", "Age", "Gender", "  Female", "  Male", "  Other", "Race", "  Asian", "  Black", "  Latino/a/x*", "  Native American", "  White", "  Multiracial*", "  Other", "Income", "  <$25,000", "  $25,001-$50,000", "  $50,001-$75,000", "  $75,001-$100,000", "  >$100,000", "  Prefer not to answer")
table_delay_unl1_clean <- c(
  paste("Study 1 (N=", demo_delay_unl1_clean$sample_size, ")", sep = ""), "",
  paste(unname(demo_delay_unl1_clean$condition["days"]), " (", unname(demo_delay_unl1_clean$condition_percent["days"]), ")", sep = ""),
  paste(unname(demo_delay_unl1_clean$condition["weeks"]), " (", unname(demo_delay_unl1_clean$condition_percent["weeks"]), ")", sep = ""),
  paste(round(demo_delay_unl1_clean$age_mean, 2), " (", demo_delay_unl1_clean$age_min, "-", demo_delay_unl1_clean$age_max, ")", sep = ""), "",
  paste(unname(demo_delay_unl1_clean$gender["Female"]), " (", unname(demo_delay_unl1_clean$gender_percent["Female"]), ")", sep = ""),
  paste(unname(demo_delay_unl1_clean$gender["Male"]), " (", unname(demo_delay_unl1_clean$gender_percent["Male"]), ")", sep = ""),
  paste(unname(demo_delay_unl1_clean$gender["Other"]), " (", unname(demo_delay_unl1_clean$gender_percent["Other"]), ")", sep = ""), "",
  paste(unname(demo_delay_unl1_clean$race["Asian"]), " (", unname(demo_delay_unl1_clean$race_percent["Asian"]), ")", sep = ""),
  paste(unname(demo_delay_unl1_clean$race["Black"]), " (", unname(demo_delay_unl1_clean$race_percent["Black"]), ")", sep = ""),
  paste(unname(demo_delay_unl1_clean$race["Latino"]), " (", unname(demo_delay_unl1_clean$race_percent["Latino"]), ")", sep = ""),
  paste(unname(demo_delay_unl1_clean$race["NativeAmerican"]), " (", unname(demo_delay_unl1_clean$race_percent["NativeAmerican"]), ")", sep = ""),
  paste(unname(demo_delay_unl1_clean$race["White"]), " (", unname(demo_delay_unl1_clean$race_percent["White"]), ")", sep = ""),
  paste(unname(demo_delay_unl1_clean$race["Multiracial"]), " (", unname(demo_delay_unl1_clean$race_percent["Multiracial"]), ")", sep = ""), 
  paste(unname(demo_delay_unl1_clean$race["Other"]), " (", unname(demo_delay_unl1_clean$race_percent["Other"]), ")", sep = ""), "",
  paste(unname(demo_delay_unl1_clean$income["<$25000"]), " (", unname(demo_delay_unl1_clean$income_percent["<$25000"]), ")", sep = ""),
  paste(unname(demo_delay_unl1_clean$income["$25000"]), " (", unname(demo_delay_unl1_clean$income_percent["$25000"]), ")", sep = ""),
  paste(unname(demo_delay_unl1_clean$income["$50000"]), " (", unname(demo_delay_unl1_clean$income_percent["$50000"]), ")", sep = ""),
  paste(unname(demo_delay_unl1_clean$income["$75000"]), " (", unname(demo_delay_unl1_clean$income_percent["$75000"]), ")", sep = ""),
  paste(unname(demo_delay_unl1_clean$income[">$100000"]), " (", unname(demo_delay_unl1_clean$income_percent[">$100000"]), ")", sep = ""),
  paste(unname(demo_delay_unl1_clean$income["Noanswer"]), " (", unname(demo_delay_unl1_clean$income_percent["Noanswer"]), ")", sep = "")
)

## Delay UNL 2 (Online) -----------------------------------------------

## Create demographics table for delay UNL 2
table_delay_unl2_clean <- c(
  paste("Study 2 (N=", demo_delay_unl2_clean$sample_size, ")", sep = ""), "",
  paste(unname(demo_delay_unl2_clean$condition["days"]), " (", unname(demo_delay_unl2_clean$condition_percent["days"]), ")", sep = ""),
  paste(unname(demo_delay_unl2_clean$condition["weeks"]), " (", unname(demo_delay_unl2_clean$condition_percent["weeks"]), ")", sep = ""),
  paste(round(demo_delay_unl2_clean$age_mean, 2), " (", demo_delay_unl2_clean$age_min, "-", demo_delay_unl2_clean$age_max, ")", sep = ""), "",
  paste(unname(demo_delay_unl2_clean$gender["Female"]), " (", unname(demo_delay_unl2_clean$gender_percent["Female"]), ")", sep = ""),
  paste(unname(demo_delay_unl2_clean$gender["Male"]), " (", unname(demo_delay_unl2_clean$gender_percent["Male"]), ")", sep = ""),
  paste(unname(demo_delay_unl2_clean$gender["Other"]), " (", unname(demo_delay_unl2_clean$gender_percent["Other"]), ")", sep = ""), "",
  paste(unname(demo_delay_unl2_clean$race["Asian"]), " (", unname(demo_delay_unl2_clean$race_percent["Asian"]), ")", sep = ""),
  paste(unname(demo_delay_unl2_clean$race["Black"]), " (", unname(demo_delay_unl2_clean$race_percent["Black"]), ")", sep = ""),
  paste(unname(demo_delay_unl2_clean$race["Latino"]), " (", unname(demo_delay_unl2_clean$race_percent["Latino"]), ")", sep = ""),
  paste(unname(demo_delay_unl2_clean$race["NativeAmerican"]), " (", unname(demo_delay_unl2_clean$race_percent["NativeAmerican"]), ")", sep = ""),
  paste(unname(demo_delay_unl2_clean$race["White"]), " (", unname(demo_delay_unl2_clean$race_percent["White"]), ")", sep = ""),
  paste(unname(demo_delay_unl2_clean$race["Multiracial"]), " (", unname(demo_delay_unl2_clean$race_percent["Multiracial"]), ")", sep = ""), 
  paste(unname(demo_delay_unl2_clean$race["Other"]), " (", unname(demo_delay_unl2_clean$race_percent["Other"]), ")", sep = ""), "",
  paste(unname(demo_delay_unl2_clean$income["<$25000"]), " (", unname(demo_delay_unl2_clean$income_percent["<$25000"]), ")", sep = ""),
  paste(unname(demo_delay_unl2_clean$income["$25000"]), " (", unname(demo_delay_unl2_clean$income_percent["$25000"]), ")", sep = ""),
  paste(unname(demo_delay_unl2_clean$income["$50000"]), " (", unname(demo_delay_unl2_clean$income_percent["$50000"]), ")", sep = ""),
  paste(unname(demo_delay_unl2_clean$income["$75000"]), " (", unname(demo_delay_unl2_clean$income_percent["$75000"]), ")", sep = ""),
  paste(unname(demo_delay_unl2_clean$income[">$100000"]), " (", unname(demo_delay_unl2_clean$income_percent[">$100000"]), ")", sep = ""),
  paste(unname(demo_delay_unl2_clean$income["Noanswer"]), " (", unname(demo_delay_unl2_clean$income_percent["Noanswer"]), ")", sep = "")
)


## Delay MTurk (Online) -----------------------------------------------

## Create demographics table for delay MTurk
table_delay_mturk_clean <- c(
  paste("Study 3 (N=", demo_delay_mturk_clean$sample_size, ")", sep = ""), "",
  paste(unname(demo_delay_mturk_clean$condition["days"]), " (", unname(demo_delay_mturk_clean$condition_percent["days"]), ")", sep = ""),
  paste(unname(demo_delay_mturk_clean$condition["weeks"]), " (", unname(demo_delay_mturk_clean$condition_percent["weeks"]), ")", sep = ""),
  paste(round(demo_delay_mturk_clean$age_mean, 2), " (", demo_delay_mturk_clean$age_min, "-", demo_delay_mturk_clean$age_max, ")", sep = ""), "",
  paste(unname(demo_delay_mturk_clean$gender["Female"]), " (", unname(demo_delay_mturk_clean$gender_percent["Female"]), ")", sep = ""),
  paste(unname(demo_delay_mturk_clean$gender["Male"]), " (", unname(demo_delay_mturk_clean$gender_percent["Male"]), ")", sep = ""),
  paste(unname(demo_delay_mturk_clean$gender["Other"]), " (", unname(demo_delay_mturk_clean$gender_percent["Other"]), ")", sep = ""), "",
  paste(unname(demo_delay_mturk_clean$race["Asian"]), " (", unname(demo_delay_mturk_clean$race_percent["Asian"]), ")", sep = ""),
  paste(unname(demo_delay_mturk_clean$race["Black"]), " (", unname(demo_delay_mturk_clean$race_percent["Black"]), ")", sep = ""),
  paste(unname(demo_delay_mturk_clean$race["Latino"]), " (", unname(demo_delay_mturk_clean$race_percent["Latino"]), ")", sep = ""),
  paste(unname(demo_delay_mturk_clean$race["NativeAmerican"]), " (", unname(demo_delay_mturk_clean$race_percent["NativeAmerican"]), ")", sep = ""),
  paste(unname(demo_delay_mturk_clean$race["White"]), " (", unname(demo_delay_mturk_clean$race_percent["White"]), ")", sep = ""),
  paste(unname(demo_delay_mturk_clean$race["Multiracial"]), " (", unname(demo_delay_mturk_clean$race_percent["Multiracial"]), ")", sep = ""), 
  paste(unname(demo_delay_mturk_clean$race["Other"]), " (", unname(demo_delay_mturk_clean$race_percent["Other"]), ")", sep = ""), "",
  paste(unname(demo_delay_mturk_clean$income["<$25000"]), " (", unname(demo_delay_mturk_clean$income_percent["<$25000"]), ")", sep = ""),
  paste(unname(demo_delay_mturk_clean$income["$25000"]), " (", unname(demo_delay_mturk_clean$income_percent["$25000"]), ")", sep = ""),
  paste(unname(demo_delay_mturk_clean$income["$50000"]), " (", unname(demo_delay_mturk_clean$income_percent["$50000"]), ")", sep = ""),
  paste(unname(demo_delay_mturk_clean$income["$75000"]), " (", unname(demo_delay_mturk_clean$income_percent["$75000"]), ")", sep = ""),
  paste(unname(demo_delay_mturk_clean$income[">$100000"]), " (", unname(demo_delay_mturk_clean$income_percent[">$100000"]), ")", sep = ""),
  paste(unname(demo_delay_mturk_clean$income["Noanswer"]), " (", unname(demo_delay_mturk_clean$income_percent["Noanswer"]), ")", sep = "")
)


## Model comparison table ------------------------------------------------

numeracy_sim_model_comparisons <- bind_cols("Model" = comparison_amt_numeracy_amt_unl1$Model, "Amount UNL 1" = exp(comparison_amt_numeracy_amt_unl1$log_BF), "Amount UNL 2" = exp(comparison_amt_numeracy_amt_unl2$log_BF), "Delay UNL 1" = exp(comparison_delay_numeracy_delay_unl1$log_BF), "Delay UNL 2" = exp(comparison_delay_numeracy_delay_unl2$log_BF), "Delay MTurk" = exp(comparison_delay_numeracy_delay_mturk$log_BF)) %>% 
  filter(`Amount UNL 1` != 1)
numeracy_itc_model_comparisons <- bind_cols("Model" = comparison_itc_numeracy_amt_unl1$Model, "Amount UNL 1" = exp(comparison_itc_numeracy_amt_unl1$log_BF), "Amount UNL 2" = exp(comparison_itc_numeracy_amt_unl2$log_BF), "Delay UNL 1" = exp(comparison_itc_numeracy_delay_unl1$log_BF), "Delay UNL 2" = exp(comparison_itc_numeracy_delay_unl2$log_BF), "Delay MTurk" = exp(comparison_itc_numeracy_delay_mturk$log_BF)) %>% 
  filter(`Amount UNL 1` != 1)
number_sim_model_comparisons <- bind_cols("Model" = comparison_amt_number_amt_unl1$Model, "Amount UNL 1" = exp(comparison_amt_number_amt_unl1$log_BF), "Amount UNL 2" = exp(comparison_amt_number_amt_unl2$log_BF), "Delay UNL 1" = exp(comparison_delay_number_delay_unl1$log_BF), "Delay UNL 2" = exp(comparison_delay_number_delay_unl2$log_BF)) %>% 
  filter(`Amount UNL 1` != 1)
number_itc_model_comparisons <- bind_cols("Model" = comparison_itc_number_amt_unl1$Model, "Amount UNL 1" = exp(comparison_itc_number_amt_unl1$log_BF), "Amount UNL 2" = exp(comparison_itc_number_amt_unl2$log_BF), "Delay UNL 1" = exp(comparison_itc_number_delay_unl1$log_BF), "Delay UNL 2" = exp(comparison_itc_number_delay_unl2$log_BF)) %>% 
  filter(`Amount UNL 1` != 1)
model_comparisons <- bind_rows(numeracy_sim_model_comparisons, numeracy_itc_model_comparisons, number_sim_model_comparisons, number_itc_model_comparisons) %>% 
  mutate(Model = str_replace_all(Model, "log_number", "log(numberline)")) %>% 
  mutate(across(where(is.numeric), ~round(.x, 2)))

save.image("framing_workspace.RData")
